# -*- coding: utf-8 -*-

import os
import logging

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, DictRepeater, SandboxStringParameter

from sandbox.projects.common.nanny import nanny


class NewsResourcesArcadiaPath(SandboxArcadiaUrlParameter):
    name = 'news_resources_arcadia_path'
    description = 'Arcadia svn url:'
    default_value = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/yweb/news/services/[SERVICE]'


class OverrideResourcePaths(DictRepeater, SandboxStringParameter):
    name = 'override_resource_paths'
    description = 'Manually override the Arcadia URL for specific file (full path to individual file)'
    required = False


def params_for_resources(resources):
    classes = []
    for resource_type in resources:
        name = resource_type.__name__
        short_name = name.lower().replace('_package', '').replace('_bundle', '')
        attrs = {
            'name': 'res_' + name.lower(),
            'description': short_name.replace('_', ' '),
            'resource_type': resource_type
        }
        ParamClass = type(name, (LastReleasedResource,), attrs)
        classes.append(ParamClass)
    return classes


#  Осторожно: задача не поддерживает сборку из веток, наложение патчей и прекоммитные проверки
#  Попробуй BuildNewsServiceConfigBundleBase2 или альтернативную сборочную задачу

class BuildNewsServiceConfigBundleBase(nanny.ReleaseToNannyTask, SandboxTask):
    """Build bundle with config files and scripts for Y.News' abstract Nanny service"""

    execution_space = 100

    environment = (environments.SvnEnvironment(),)

    input_parameters = [
        NewsResourcesArcadiaPath,
        OverrideResourcePaths
    ]

    NEWS_CONFIG_RESOURCES = {}

    def get_news_resources_path(self):
        return self.path('news_resources')

    def on_execute(self):
        logging.info('Get dir with news resources from')
        news_resources = self.get_news_resources_path()
        custom_resources = self.path('custom_resources')
        news_svn_url = Arcadia.parse_url(self.ctx[NewsResourcesArcadiaPath.name])
        Arcadia.export(self.ctx[NewsResourcesArcadiaPath.name], news_resources)

        custom_urls = self.ctx['override_resource_paths']

        if not isinstance(custom_urls, dict):
            custom_urls = None

        if custom_urls is not None:
            os.mkdir(custom_resources)
            for file_name, arc_path in custom_urls.iteritems():
                Arcadia.export(arc_path, os.path.join(custom_resources, file_name))

        logging.info('Copy files from {} dir and save them as resources')
        for file_name, resource_type in self.NEWS_CONFIG_RESOURCES.iteritems():
            current_svn_url = news_svn_url
            source_file_path = os.path.join(news_resources, file_name)
            if custom_urls is not None:
                if file_name in custom_urls.keys():
                    source_file_path = os.path.join(custom_resources, file_name)
                    current_svn_url = Arcadia.parse_url(custom_urls.get(file_name))
            destination_file_path = self.path(file_name)
            copy_path(source_file_path, destination_file_path)
            if not os.path.exists(source_file_path):
                raise SandboxTaskFailureError('Cannot find file {} in {}'.format(
                    file_name, current_svn_url
                ))
            self.create_resource(
                description='Infrastructure file {} for Nanny service, version {}'.format(
                    file_name, current_svn_url.revision),
                resource_path=destination_file_path,
                resource_type=resource_type,
                arch="any"
            )
