import os
import tempfile
import re

import platform

from sandbox.projects import resource_types
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import process

url_re = re.compile('^[a-z0-9+_-]+://')


def split_reqs(file):
    reqs = []
    urls = []
    with open(file) as fd:
        for line in fd:
            line = line.rstrip()
            (urls if url_re.match(line) else reqs).append(line)

    with open(file, 'w') as fd:
        for r in reqs:
            fd.write(r + "\n")

    return urls


class RequirementsSvnPath(parameters.SandboxSvnUrlParameter):
    name = 'requirements_svn_path'
    default_value = ''
    description = "SVN path to file with requirements list"
    required = True


class UseSandboxGCCParameter(parameters.SandboxBoolParameter):
    name = 'use_sandbox_gcc'
    description = 'Use Sandbox GCC'
    default_value = True


class BuildNewsVirtualenv(task.SandboxTask):
    type = 'BUILD_NEWS_VIRTUALENV'
    input_parameters = [
        RequirementsSvnPath,
        UseSandboxGCCParameter,
    ]

    result_archive_name = 'news-virtualenv.tar.gz'
    resource_description = "News python virtualenv"
    resource_type = resource_types.NEWS_VIRTUALENV

    def on_execute(self):
        reqs_filename = self.path('reqs.txt')
        reqs_svn_url = self.ctx[RequirementsSvnPath.name]
        rev = svn.Arcadia.get_revision(reqs_svn_url)
        svn.Arcadia.export(reqs_svn_url, reqs_filename, revision=rev)
        urls = split_reqs(reqs_filename)
        with environments.VirtualEnvironment() as venv:
            venv.pip('--use-wheel -r {}'.format(reqs_filename))
            for murl in urls:
                name = os.path.basename(murl)
                tempdir = tempfile.mkdtemp(dir=self.path(), prefix=name)
                svn.Arcadia.export(murl, tempdir, revision=rev)
                process.run_process([venv.executable, 'setup.py', 'install'], work_dir=tempdir, log_prefix='install-%s' % name)
            self.result_dir_name = venv.root_dir
            with open(os.path.join(venv.root_dir, '.revision'), 'w') as f:
                f.write(str(rev))
            venv.pack_bundle(self.result_archive_name)

        attrs = {'platform': platform.platform()}

        self.create_resource(description=self.resource_description,
                             resource_path=self.result_archive_name,
                             resource_type=self.resource_type,
                             attributes=attrs)


__Task__ = BuildNewsVirtualenv
