# -*- coding: utf-8 -*-
import tarfile
import os
import shutil

from sandbox.projects.news.resources import RELEASERS as rs
import sandbox.projects.common.gnupg
from sandbox.projects.common.nanny import nanny

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.misc as ctm


class SportApiPackage(sdk2.Resource):
    """
    deb пакет сo спортивным апи
    """
    releasable = True
    auto_backup = True
    releasers = rs


class BuildSportApi(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Таск для сборки пакета спортивного api (yandex-sport-api.deb)
    """
    class Parameters(sdk2.Task.Parameters):
        max_restarts = 1
        kill_timeout = 1 * 3600

        git_repo = sdk2.parameters.String('Git repository to build',
                                          required=False)

        code_archive = sdk2.parameters.Resource('Node API archive', required=False)

        branch = sdk2.parameters.String('Git branch',
                                        default='master',
                                        required=False)

        vault_items_owner = sdk2.parameters.String('Vault items owner',
                                                   default='NEWS_PROD',
                                                   required=True)

        robot_name = sdk2.parameters.String('Robot\'s name to access GitHub and sign deb packages. Используется как префикс для ключей в Vault',
                                            default='robot-sport-robot',
                                            required=True)

        _container = sdk2.parameters.Container("Build environment container resource",
                                               default_value=619842771,
                                               required=True
                                               )

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64

    def on_execute(self):
        username = self.Parameters.robot_name
        email = "news-dev@yandex-team.ru"

        build_dir = "build"
        if os.path.exists(build_dir):
            shutil.rmtree(build_dir)
        os.makedirs(build_dir)
        os.chdir(build_dir)

        os.environ["LC_ALL"] = "ru_RU.UTF-8"
        os.environ["LANG"] = "en_US.UTF8"
        os.environ['PATH'] = "/opt/nodejs/0.10/bin:" + os.environ['PATH']

        with sdk2.ssh.Key(self, self.Parameters.vault_items_owner, self.Parameters.robot_name + "-ssh"):
            if self.Parameters.code_archive:
                code = sdk2.ResourceData(self.Parameters.code_archive)
                with tarfile.open(str(code.path), 'r') as f:
                    f.extractall()
                    os.chdir("api")
            else:
                with sdk2.helpers.ProcessLog(self, logger="git") as pl:
                    sp.check_call(['git', 'clone', self.Parameters.git_repo], stderr=pl.stderr, stdout=pl.stdout)
                    os.chdir("api")
                    sp.check_call(['git', 'config', 'user.name', username], stderr=pl.stderr, stdout=pl.stdout)
                    sp.check_call(['git', 'config', 'user.email', email], stderr=pl.stderr, stdout=pl.stdout)
                    sp.check_call(['git', 'checkout', self.Parameters.branch], stderr=pl.stderr, stdout=pl.stdout)
                    pl.logger.info('Git clone finished successfully')

            os.environ["DEBFULLNAME"] = username
            os.environ["DEBEMAIL"] = email
            with sdk2.helpers.ProcessLog(self, logger="build") as pl:
                with sandbox.projects.common.gnupg.GpgKey2(self.Parameters.vault_items_owner, self.Parameters.robot_name + '-gpg-private', self.Parameters.robot_name + '-gpg-public'):
                    sp.check_call(['npm', 'set', 'registry', 'https://npm.yandex-team.ru'], stderr=pl.stderr, stdout=pl.stdout)
                    sp.check_call(['npm', 'install'], stderr=pl.stderr, stdout=pl.stdout)
                    sp.check_call(['./node_modules/.bin/grunt', 'release', '--quiet=true', '--nopublish=true'], stderr=pl.stderr, stdout=pl.stdout)
                pl.logger.info('Build finished successfully')

            changelog = ""
            with open("debian/changelog") as fd:
                counter = 0
                for line in fd:
                    if line.startswith("yandex-sport-api"):
                        counter += 1
                    if counter > 1:
                        break
                    changelog += line
            self.Parameters.description += "\n\nChangelog:\n" + changelog

        os.chdir("..")
        for f in [f for f in os.listdir(".") if os.path.isfile(f)]:
            if str(f).startswith("yandex-sport-api") and str(f).endswith(".deb"):
                file_name = str(self.path(f))
                shutil.move(f, file_name)
                SportApiPackage(self, "Sport API package", file_name)
                break
