# -*- coding: utf-8 -*-
import os
import shutil

from sandbox.projects.news.resources import RELEASERS as rs
import sandbox.projects.common.gnupg
from sandbox.projects.common.nanny import nanny
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp


class SportBackendBuildContainer(sdk2.service_resources.LxcContainer):
    """
    LXC контейнер со окружением для сборки бэкенда спортивного апи
    """
    releasable = True
    auto_backup = True
    releasers = rs


class SportApiAdminPackage(sdk2.Resource):
    """
    deb пакет с админкой спортивного апи
    """
    releasable = True
    auto_backup = True
    releasers = rs


class SportApiImporterPackage(sdk2.Resource):
    """
    deb пакет с импортерами спортивного апи
    """
    releasable = True
    auto_backup = True
    releasers = rs


class BuildSportBackend(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Таск для сборки пакетов спортивного бэкенда: yandex-sport-admin.deb и yandex-sport-importer.deb
    """
    class Parameters(sdk2.Task.Parameters):
        max_restarts = 1
        kill_timeout = 1 * 3600

        git_repo = sdk2.parameters.String('Git repository to build',
                                          default='git@github.yandex-team.ru:sport/sochi-backend.git',
                                          required=True)

#        custom_branch = sdk2.parameters.Bool("Build from custom branch")
#        with custom_branch.value[True]:
        branch = sdk2.parameters.String('Git branch',
                                        default='master',
                                        required=True)

#        with sdk2.parameters.String("Release type", required=True, multiline=True) as release_type:
#            release_type.values.patch = "patch"
#            release_type.values.minor = "minor"
#            release_type.values.major = "major"

        vault_items_owner = sdk2.parameters.String('Vault items owner',
                                                   default='NEWS_PROD',
                                                   required=True)

        robot_name = sdk2.parameters.String('Robot\'s name to access GitHub and sign deb packages. Используется как префикс для ключей в Vault',
                                            default='robot-sport-robot',
                                            required=True)

        new_image = sdk2.parameters.Bool(
            'New image used',
            default=False,
        )

        _container = sdk2.parameters.Container('Build environment container resource',
                                               default_value=612743707,
                                               resource_type=SportBackendBuildContainer,
                                               required=True
                                               )

    def on_execute(self):
        username = self.Parameters.robot_name
        email = "news-dev@yandex-team.ru"

        build_dir = "build"
        if os.path.exists(build_dir):
            shutil.rmtree(build_dir)
        os.makedirs(build_dir)
        os.chdir(build_dir)

        os.environ["LC_ALL"] = "ru_RU.UTF-8"
        os.environ["LANG"] = "en_US.UTF8"

        with sdk2.ssh.Key(self, self.Parameters.vault_items_owner, self.Parameters.robot_name + "-ssh"):
            with sdk2.helpers.ProcessLog(self, logger="git") as pl:
                sp.check_call(['git', 'clone', self.Parameters.git_repo], stderr=pl.stderr, stdout=pl.stdout)
                os.chdir("sochi-backend")
                sp.check_call(['git', 'config', 'user.name', username], stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call(['git', 'config', 'user.email', email], stderr=pl.stderr, stdout=pl.stdout)
                sp.check_call(['git', 'checkout', self.Parameters.branch], stderr=pl.stderr, stdout=pl.stdout)
                pl.logger.info('Git clone finished successfully')

            os.environ["DEBFULLNAME"] = username
            os.environ["DEBEMAIL"] = email
            with sdk2.helpers.ProcessLog(self, logger="build") as pl, \
                sandbox.projects.common.gnupg.GpgKey2(
                self.Parameters.vault_items_owner,
                self.Parameters.robot_name + '-gpg-private',
                self.Parameters.robot_name + '-gpg-public'
            ):
                #                if self.Parameters.custom_branch:
                sp.check_call(['scripts/snapshot_changelog.sh'], stderr=pl.stderr, stdout=pl.stdout)
                if self.Parameters.new_image:
                    sp.check_call(
                        ['gpgconf', '--list-dir', 'agent-extra-socket'],
                        stderr=pl.stderr,
                        stdout=pl.stdout
                    )
                    sp.check_call(
                        ['gbp', 'buildpackage', '--git-ignore-new', '--git-debian-branch=' + self.Parameters.branch],
                        stderr=pl.stderr,
                        stdout=pl.stdout
                    )
                else:
                    sp.check_call(
                        ['git-buildpackage', '--git-ignore-new', '--git-debian-branch=' + self.Parameters.branch],
                        stderr=pl.stderr,
                        stdout=pl.stdout
                    )
#                    sp.check_call(['git', 'commit', 'debian/changelog', '-m', 'new release'], stderr=pl.stderr, stdout=pl.stdout)
#                    sp.call(['git', 'push', 'origin', self.Parameters.branch], stderr=pl.stderr, stdout=pl.stdout)
#                else:
#                    sp.check_call(['scripts/release.sh', self.Parameters.release_type], stderr=pl.stderr, stdout=pl.stdout)
#                    sp.check_call(['git-buildpackage', '--git-ignore-new', '--git-debian-branch=' + self.Parameters.branch], stderr=pl.stderr, stdout=pl.stdout)

#                    sp.check_call(['git', 'push', '--tags', '--quiet', 'origin',  self.Parameters.branch], stderr=pl.stderr, stdout=pl.stdout)

            changelog = ""
            with open("debian/changelog") as fd:
                counter = 0
                for line in fd:
                    if line.startswith("yandex-sport"):
                        counter += 1
                    if counter > 1:
                        break
                    changelog += line
#            self.Parameters.description += "\n\nChangelog:\n" + changelog

        os.chdir("..")
        for f in [f for f in os.listdir(".") if os.path.isfile(f) and f.endswith('.deb')]:
            if str(f).startswith("yandex-sport-admin"):
                file_name = str(self.path(f))
                shutil.move(f, file_name)
                SportApiAdminPackage(self, "New sport admin package", file_name)
            if str(f).startswith("yandex-sport-importer"):
                file_name = str(self.path(f))
                shutil.move(f, file_name)
                SportApiImporterPackage(self, "New sport importers package", file_name)
