import requests
import json

from sandbox.projects.common.juggler import jclient
from sandbox import sdk2

Locations = [
    'sas',
    'man',
    'vla'
]


class CheckNewsItsFlagsCount(sdk2.Task):
    """ This task checks if number of ITS flags less than N  """

    class Parameters(sdk2.Task.Parameters):
        description = "This task checks if number of ITS flags less than N"
        max_flags_count = sdk2.parameters.Integer(
            "Max allowed number of ITS flags",
            required=True,
            default_value=10
        )

    def get_its_flags_from_location(self, path):
        request = requests.get('http://its.yandex-team.ru/v1/values/' + path + '/', headers={
            'Accept': 'application/json',
            'Content-Type': 'application/json; charset=utf-8',
            'Authorization': 'OAuth ' + sdk2.Vault.data('NEWS', 'robot_ynews_nanny_oauth_token')
        })

        data = json.loads(request.text)
        assert data["path"] == path
        assert "user_value" in data

        return json.loads(data["user_value"])

    def get_its_flags(self, path):
        first_location = True
        prev_flags = None
        for location in Locations:
            flags = self.get_its_flags_from_location(path.format(location))
            if not first_location and flags != prev_flags:
                return flags, False
            prev_flags = flags
            first_location = False

        return flags, True

    def counter(self, item, count):
        for key, value in item.iteritems():
            if isinstance(value, dict):
                count = self.counter(value, count)
            else:
                count += 1
        return count

    def on_execute(self):
        ok_consistency = True
        descr_consistency_template = 'ITS flags differ in different locations: {}'
        descr_consistency = ""

        ok_count = True
        descr_count_template = 'Too many ITS flags: {}'
        descr_count = ""

        for path in ["newsupper/{}/News/common_news_flags", "newsupper/{}/Personalization/news_personalization_default_flags"]:
            flags, ok = self.get_its_flags(path)
            if not ok:
                descr_consistency = descr_consistency_template.format(path).format("<location>")
            ok_consistency &= ok

            count = self.counter(flags, 0)
            self.set_info("Number of its flags for {} is {}".format(path, count).format("<location>"))
            ok = (count < self.Parameters.max_flags_count)
            if not ok:
                descr_count = descr_count_template.format(path).format("<location>")
            ok_count &= ok

        jclient.send_events_to_juggler(
            'check_news_its_flags_count',
            'news_its_flags_count_alert',
            'OK' if ok_count else 'CRIT',
            'Number of ITS flags is OK' if ok_count else descr_count,
        )

        jclient.send_events_to_juggler(
            'check_news_its_flags_count',
            'news_its_flags_locations_alert',
            'OK' if ok_consistency else 'CRIT',
            'ITS flags is the same in different locations' if ok_consistency else descr_consistency,
        )
