# -*- coding: utf-8 -*-

import os.path
import requests
import logging
import datetime

from sandbox.projects.common.juggler.jclient import to_juggler_service_name
from sandbox.projects.common.nanny.nanny import NannyClient
from sandbox.sandboxsdk.svn import Arcadia
from sandbox import sdk2
from sandbox.common import rest as sandbox_api

from yaml import load as yaml_load

NewsServicesCategories = [
    '/search-news',
    '/news',
    '/sepe/priemka/search/stable',
    '/search/sepe/app_host/news'
]

NewsPriemkaServicesCategories = [
    '/sepe/priemka/search/stable',
]


class CheckNewsNannyTicketsIntegration(sdk2.Task):
    """
        Проверяет наличие Tickets Integration на всех ресурсах сервисов новостей
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4096
        disk_space = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    def send_events_to_juggler(self, host, service, status, description, tags):
        try:
            reply = requests.post("http://juggler-push.search.yandex.net/events", json={
                "source": "sandbox",
                "events": [{
                    "host": host,
                    "service": service,
                    "status": status,
                    "description": description,
                    "tags": [tags]
                }]
            })
            event_status = reply.json()["events"][0]
            if event_status["code"] != 200:
                raise Exception(event_status["error"])
        except Exception:
            logging.exception("unable to get frontend nodes from juggler")

    def parse_sandbox_timestamp(self, s):
        for fmt in ('%Y-%m-%dT%H:%M:%S.%fZ', '%Y-%m-%dT%H:%M:%SZ'):
            try:
                return datetime.datetime.strptime(s, fmt)
            except ValueError:
                pass

    def check_resource_ttl(self, resource):
        resource_id = ''
        if 'resource_id' not in resource:
            # TODO: Get resource from task
            return True
        else:
            resource_id = resource['resource_id']
        attrs = self.sandbox_client.resource[resource_id].attribute.read()

        resource = self.sandbox_client.resource[resource_id].read()
        if resource['time']['expires'] is not None:
            to_expire = self.parse_sandbox_timestamp(resource['time']['expires']) - self.now
            if to_expire > self.expire_delta:
                return True

        for a in attrs:
            if a['name'] == 'ttl':
                return a['value'] == 'inf'
        return False

    def send_ti_ignored(self, service_name, reason, is_prod):
        self.send_events_to_juggler(
            to_juggler_service_name(service_name),
            'news_nanny_ti_alert',
            'OK',
            'Service is ignored ({})'.format(reason),
            'news_nanny_resources_ti_alert_prod' if is_prod else 'news_nanny_resources_ti_alert_priemka'
        )

    def send_ttl_ignored(self, service_name, reason, is_prod):
        self.send_events_to_juggler(
            to_juggler_service_name(service_name),
            'news_nanny_ttl_alert',
            'OK',
            'Service is ignored ({})'.format(reason),
            'news_nanny_resources_ttl_alert_prod' if is_prod else 'news_nanny_resources_ttl_alert_priemka'
        )

    def check_service(self, service_name, ignored_resources, ignore_resources_ttl, global_ignore, is_prod=True):
        current_state = self.nanny_client.get_service_current_state(service_name)
        if current_state['content']['summary']['value'] == 'OFFLINE':
            logging.info('Service {} is offline'.format(service_name))
            self.send_ttl_ignored(service_name, "offline", is_prod)
            self.send_ti_ignored(service_name, "offline", is_prod)
            return

        runtime_attrs = self.nanny_client.get_service_runtime_attrs(service_name)
        info_attrs = self.nanny_client.get_service_info_attrs(service_name)
        tickets_integration = info_attrs['content']['tickets_integration']
        failed_resources = []
        resources_with_short_ttl = []
        for resource in runtime_attrs['content']['resources']['sandbox_files']:
            # Check ttl of the resource
            if resource['resource_type'] not in ignore_resources_ttl and resource['resource_type'] not in global_ignore['ttl']:
                if not self.check_resource_ttl(resource):
                    resources_with_short_ttl.append(resource['resource_type'])

            # Check is TI check for resource ignored or not
            if resource['resource_type'] in ignored_resources or resource['resource_type'] in global_ignore['ti']:
                logging.info('Ignore resource {} of service {}'.format(resource, service_name))
                continue

            found = False
            for ticket in tickets_integration['service_release_rules']:
                resource_type_found = 'sandbox_resource_type' not in ticket or ticket['sandbox_resource_type'] == '' or ticket['sandbox_resource_type'] == resource['resource_type']
                task_type_found = 'sandbox_task_type' not in ticket or ticket['sandbox_task_type'] == '' or ticket['sandbox_task_type'] == resource['task_type']
                if resource_type_found and task_type_found:
                    found = True

            if not found:
                failed_resources.append(resource['resource_type'])

        if not resources_with_short_ttl:
            self.send_events_to_juggler(
                to_juggler_service_name(service_name),
                'news_nanny_ttl_alert',
                'OK',
                'All resources have infinite ttl',
                'news_nanny_resources_ttl_alert_prod' if is_prod else 'news_nanny_resources_ttl_alert_priemka'
            )
        else:
            self.send_events_to_juggler(
                to_juggler_service_name(service_name),
                'news_nanny_ttl_alert',
                'CRIT' if is_prod else 'WARN',
                'Resources {} have finite ttl'.format(resources_with_short_ttl),
                'news_nanny_resources_ttl_alert_prod' if is_prod else 'news_nanny_resources_ttl_alert_priemka'
            )

        if not failed_resources:
            self.send_events_to_juggler(
                to_juggler_service_name(service_name),
                'news_nanny_ti_alert',
                'OK',
                'All resources have tickets integration',
                'news_nanny_resources_ti_alert_prod' if is_prod else 'news_nanny_resources_ti_alert_priemka'
            )
        else:
            self.send_events_to_juggler(
                to_juggler_service_name(service_name),
                'news_nanny_ti_alert',
                'WARN',
                'Resources {} doesn\'t have tickets integration'.format(failed_resources),
                'news_nanny_resources_ti_alert_prod' if is_prod else 'news_nanny_resources_ti_alert_priemka'
            )

    def load_ignore_list(self):
        self.__conf_path = os.path.abspath('config')
        arcadia_url_splited = self.Parameters.arcadia_url.split('@')
        arcadia_conf_path = arcadia_url_splited[0] + '/yweb/news/config/nanny_ti_ignore.yaml'
        logging.info('Try to get conf from {}'.format(arcadia_conf_path))
        Arcadia.export(arcadia_conf_path, self.__conf_path)

        ignore_ti = {}
        ignore_ttl = {}
        global_ignore = {'ti': [], 'ttl': []}
        config = yaml_load(open(self.__conf_path).read())
        logging.info('conf: {}'.format(config))

        for service_name in config.get('ignore_ti', []):
            ignore_ti[service_name] = config['ignore_ti'][service_name]

        for service_name in config.get('ignore_ttl', []):
            ignore_ttl[service_name] = config['ignore_ttl'][service_name]

        for resource_name in config.get('global_ignore_ti', []):
            global_ignore['ti'].append(resource_name)

        for resource_name in config.get('global_ignore_ttl', []):
            global_ignore['ttl'].append(resource_name)

        return ignore_ti, ignore_ttl, global_ignore

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.String('Arcadia url')
        expire_to_alert = sdk2.parameters.Integer('If resource will expire sooner than N days from now', required=True, default=10)

    def on_execute(self):
        nanny_oauth_token = sdk2.Vault.data('NEWS', 'nanny_oauth_token')
        self.nanny_client = NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=nanny_oauth_token)

        self.sandbox_client = sandbox_api.Client(base_url='https://sandbox.yandex-team.ru/api/v1.0')
        self.now = datetime.datetime.utcnow()
        self.expire_delta = datetime.timedelta(days=self.Parameters.expire_to_alert)

        ignore_ti, ignore_ttl_check, global_ignore = self.load_ignore_list()

        for category in NewsServicesCategories:
            services = self.nanny_client.list_services_by_category(category)['result']
            for service in services:
                service_name = service['_id']
                is_testing = 'r1' in service_name or 'test' in service_name or 'priemka' in service_name or 'sink' in service_name
                logging.info('Service {} is_prod: {}'.format(service_name, not is_testing))
                if service_name in ignore_ti:
                    if ignore_ti[service_name]:
                        logging.info('Process service {} and ignore resources {}'.format(service_name, ignore_ti[service_name]))
                        self.check_service(service_name, ignore_ti[service_name], ignore_ttl_check.get(service_name, []), global_ignore, not is_testing)
                    else:
                        self.send_ti_ignored(service_name, "service ignored", True)
                        logging.info('Ignore service {}'.format(service_name))
                else:
                    logging.info('Process service {}'.format(service_name))
                    self.check_service(service_name, [], ignore_ttl_check.get(service_name, []), global_ignore, not is_testing)

        for category in NewsPriemkaServicesCategories:
            services = self.nanny_client.list_services_by_category(category)['result']
            for service in services:
                service_name = service['_id']
                if service_name in ignore_ti:
                    if ignore_ti[service_name]:
                        logging.info('Process service {} and ignore resources {}'.format(service_name, ignore_ti[service_name]))
                        self.check_service(service_name, ignore_ti[service_name], ignore_ttl_check.get(service_name, []), global_ignore, False)
                    else:
                        self.send_ti_ignored(service_name, "service ignored", False)
                        logging.info('Ignore service {}'.format(service_name))
                else:
                    logging.info('Process service {}'.format(service_name))
                    self.check_service(service_name, [], ignore_ttl_check.get(service_name, []), global_ignore, False)
