# -*- coding: utf-8 -*-
import logging

from sandbox.projects.common import error_handlers as eh

from sandbox.projects.websearch.basesearch.CompareBasesearchPerformance import CompareBasesearchPerformance, MaxDiff, MaxDiffPercent, CompareType


class CompareNewsPerformance(CompareBasesearchPerformance):
    """
    **Описание**
        Сравнение производительности двух сборок базового поиска.
        Также работает для визарда.

    **Ресурсы**
        *Необходимые для запуска ресурсы и параметры*
            * test results 1 - идентификатор задачи с тестированием производительности первой сборки
            * test results 2 - идентификатор задачи с тестированием производительности второй сборки
            * Number of top results taken into account - число результатов, по которым будет идти сравнение
            * max diff (request/sec) - максимальная допустимая разница
            * Percentage (%)
            * Comparison type
            * Check on intervals
        *Создаваемые ресурсы*
            * BASESEARCH_PERFORMANCE_COMPARE_RESULT
    """

    type = 'COMPARE_NEWS_PERFORMANCE'

    input_parameters = (
        CompareBasesearchPerformance.input_parameters
    )

    def _check_performance(self, stats_task1, stats_task2, top_results_number):
        """
            Сравнивает производительность
        """
        try:
            top1, avg1 = self._top_results(stats_task1['requests_per_sec'], top_results_number)
            median1 = self._median_results(stats_task1['requests_per_sec'])
        except KeyError:
            eh.check_failed(
                "No rps data aggregated, maybe test results 1 are too old and resources expired"
            )
        try:
            top2, avg2 = self._top_results(stats_task2['requests_per_sec'], top_results_number)
            median2 = self._median_results(stats_task2['requests_per_sec'])
        except KeyError:
            eh.check_failed(
                "No rps data aggregated, maybe test results 2 are too old and resources expired"
            )
        diff = median1 - median2

        if self.ctx[CompareType.name] == 'rps':
            compare_result = (abs(diff) <= self.ctx[MaxDiff.name])
            self.set_info("RPS diff: {:+.1f}".format(diff))
        elif self.ctx[CompareType.name] == 'percent':
            if max(top1) < 1.0:
                logging.info('Diff failed -- First task RPSes are too low')
                compare_result = False
            else:
                diff_percent = 100.0 * diff / max(top1)
                logging.info('Diff per cent: %s', diff_percent)
                compare_result = (abs(diff_percent) <= self.ctx[MaxDiffPercent.name])
                self.set_info("RPS diff: {:+.1f} (<strong>{:+.2f}%</strong>)".format(diff, diff_percent), do_escape=False)
        else:
            eh.fail("Unknown compare type: {}".format(self.ctx[CompareType.name]))
        self.set_info("Verdict: <strong>{}</strong>".format("no significant difference" if compare_result else "significant difference"), do_escape=False)

        stats = top1, avg1, top2, avg2, diff
        return compare_result, stats

    @staticmethod
    def _median_results(results):
        eh.verify(len(results) >= 1, 'Test results should contain at least one results')
        return sorted(results)[len(results) // 2]


__Task__ = CompareNewsPerformance
