# -*- coding: utf-8 -*-

import hashlib

from sandbox.projects.common.nanny import nanny
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import environments


class AdvWidgetClientsFile(sdk2.Resource):
    """Advwidget clients list"""
    releasable = True
    releasers = ["NEWS"]
    revision = sdk2.parameters.String("Revision")
    ttl = 'inf'


class DownloadAdvWidgetClientsTable(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Download table from YT to a file, compare with "previous" yt table and
        create resource if table has been updated
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        environments = [environments.PipEnvironment("yandex-yt")]
        cores = 2
        disk_space = 1 * 1024  # 1Gb

    class Parameters(sdk2.Task.Parameters):
        description = "Download table from YT to a file"
        kill_timeout = 3 * 3600  # 3 hours

        yt_cluster_name1 = sdk2.parameters.String("YT Cluster name 1", default='arnold', required=True)
        yt_path1 = sdk2.parameters.String("YT table path 1", default='//home/partner/domains_for_news', required=True)

        yt_cluster_name2 = sdk2.parameters.String("YT Cluster name 2", default='hahn', required=True)
        yt_path2 = sdk2.parameters.String("YT table path 2", default='//home/partner/domains_for_news', required=True)

        yt_token_name = sdk2.parameters.String(
            "Vault secret name with YT token vault_owner:vault_key",
            default='NEWS:yt_token',
            required=True
        )
        released_attr = sdk2.parameters.String("Released attribute name to look at previous stable revision (stable, prestable, etc)", default=None, required=False)

    def previous_yt_table_version(self, attribute):
        if attribute is None:
            return None

        resid = AdvWidgetClientsFile.find(attrs=dict(released=attribute)).first()

        if resid is None:
            return None

        return sdk2.Resource[resid].revision

    def main_work(self, yt_proxy, yt_table):
        import yt.wrapper

        (owner, key) = self.Parameters.yt_token_name.split(':')
        yt_token = sdk2.Vault.data(owner, key)

        client = yt.wrapper.YtClient(proxy=yt_proxy, token=yt_token)

        previous_revision = self.previous_yt_table_version(self.Parameters.released_attr)
        rows = [r for r in client.read_table(yt_table, format="json", raw=True)]
        ls = [hashlib.md5(l).hexdigest() for l in rows]
        revision = hashlib.md5("".join(ls)).hexdigest()

        if revision == previous_revision:
            self.Parameters.description = \
                'Table {2}:{3} will not downloaded: ({0} == revision parameter {1})'.format(revision, previous_revision, yt_proxy, yt_table)
            return

        resource = AdvWidgetClientsFile(
            self,
            "Advwidget clients list",
            "advertising_widget_clients_hosts.json",
            revision=revision
        )
        resource_data = sdk2.ResourceData(resource)

        with open(str(resource_data.path.absolute()), 'w') as f:
            f.writelines(rows)

        self.Context.downloaded_resources = {'advertising_widget_clients_hosts': resource.id}

    def on_execute(self):
        try:
            self.main_work(self.Parameters.yt_cluster_name1, self.Parameters.yt_path1)
        except:
            self.main_work(self.Parameters.yt_cluster_name2, self.Parameters.yt_path2)
        return
