import subprocess
import os
from sandbox import common
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.news.resources import NEWS_RT_VH_VISITS
from sandbox.projects.common.juggler import jclient
from sandbox.projects.common.nanny import nanny
import sandbox.common.types.task as ctt
import logging
import json


class DownloadNewsRtVhVisits(nanny.ReleaseToNannyTask2, sdk2.Task):
    """ Task downloads dump of ydb-table news_rt_vh_visits and releases it to slaves """

    class Parameters(sdk2.Task.Parameters):
        description = "Task downloads dump of ydb-table news_rt_vh_visits and releases it to slaves"

        ydb_endpoint = sdk2.parameters.String(
            'YDB endpoint',
            default_value="ydb-ru.yandex.net:2135",
            required=True
        )
        ydb_database = sdk2.parameters.String(
            'YDB database',
            default_value="/ru/news/production/video-visits",
            required=True
        )
        ydb_token = sdk2.parameters.YavSecret(
            'YDB token',
            default_value="sec-01eaz1354rv2f4xnyaq4bjh07b",
            required=True
        )
        script_binary = sdk2.parameters.Resource(
            'Script binary from news/sandbox/rt_vh_visits',
            required=True
        )

    def get_path(self, file_name):
        path = os.path.join(os.path.dirname(__file__), file_name)
        if common.import_hook.inside_the_binary():
            from library.python import resource
            resource_data = resource.find(path)
            if resource_data is not None:
                with open(file_name, 'w') as f:
                    f.write(resource_data)
                    return os.path.abspath(file_name)
        return path

    def on_execute(self):
        env_vars = os.environ.copy()
        env_vars["YDB_TOKEN"] = self.Parameters.ydb_token.data()['robot-ynews-ydb-token']

        with sdk2.helpers.ProcessLog(self, logger="script-log") as pl:
            subprocess.check_call(
                [
                    str(sdk2.ResourceData(self.Parameters.script_binary).path),
                    self.Parameters.ydb_endpoint,
                    self.Parameters.ydb_database,
                ],
                stdout=pl.stdout,
                stderr=pl.stdout,
                env=env_vars
            )

        resource = NEWS_RT_VH_VISITS(self, "News video visits", "data.json")
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)

        need_release = True

        old_video_visits = sdk2.Resource.find(
            resource_type=NEWS_RT_VH_VISITS,
            attrs={'released': 'stable'}
        ).order(-sdk2.Resource.id).first()
        logging.info("old video visits: " + str(old_video_visits))

        with open("data.json", "r") as f:
            new_json = json.load(f)

        ok = True
        if len(new_json) == 0:
            ok = False

        try:
            jclient.send_events_to_juggler(
                "news-rt-vh-visits-scheduler",
                "news-rt-vh-visits-scheduler",
                "OK" if ok else "CRIT",
                "Scheduler with rt vh visits: https://sandbox.yandex-team.ru/scheduler/24866/view"
            )
        except Exception:
            self.set_info('Failed to send event to juggler')
            logging.error('Failed to send event to juggler')

        if not ok:
            return

        if old_video_visits:
            with open(str(sdk2.ResourceData(old_video_visits).path), "r") as f:
                old_json = json.load(f)

            if old_json == new_json:
                logging.info("old == new, do not release")
                need_release = False

        if need_release:
            nanny.ReleaseToNannyTask2.on_release(self, dict(
                releaser=self.author,
                release_status=ctt.ReleaseStatus.STABLE,
                release_subject="news rt vh visits",
                email_notifications=dict(to=[], cc=[]),
                release_comments="news video visits",
            ))
