# -*- coding: utf-8 -*-

import json
import hashlib
import logging
from urllib2 import urlopen, Request

from sandbox.sandboxsdk import parameters
from sandbox import sdk2

from sandbox.projects.common import apihelpers
from sandbox.projects.common import link_builder
from sandbox.projects.news.resources import NEWS_THEMES_TAGS_TO_COMPONENTS
from sandbox.projects.NewsAutoreleasableTask import NewsAutoreleasableTask
from sandbox.projects.NewsAutoreleasableTask import Params as NewsAutoreleasableTaskParams


class Params(object):
    class ThemesAdminUrl(parameters.SandboxStringParameter):
        name = 'themes_admin_url'
        description = 'Themes admin url'
        default_value = 'http://themes.news-viewers.yandex-team.ru/'

    class ApiPath(parameters.SandboxStringParameter):
        name = 'api_path'
        description = 'API path'
        default_value = 'api/tags_to_components'

    class OutputFileName(parameters.SandboxStringParameter):
        name = 'output_file_name'
        description = 'Output file name'
        default_value = 'tags_to_components.tsv'

    params = [ThemesAdminUrl, ApiPath, OutputFileName]


class TagMapping(object):
    __slots__ = ["url", "component", "tag"]

    def __init__(self, row):
        self.url = row["url"]
        self.component = row["component"]
        self.tag = row["tag"]

    def to_line(self):
        return "{}\t{}\n".format(self.tag, self.component)

    def to_dict(self):
        return {
            "url": self.url,
            "component": self.component,
            "tag": self.tag,
        }


class DownloadNewsThemesTagsToComponents(NewsAutoreleasableTask):
    """"""

    type = "DOWNLOAD_NEWS_THEMES_TAGS_TO_COMPONENTS"
    cores = 1
    execution_space = 4096  # 4 GB

    input_parameters = NewsAutoreleasableTaskParams.params + Params.params

    def do_execute(self):
        admin_url = self.ctx.get(Params.ThemesAdminUrl.name)
        api_path = self.ctx.get(Params.ApiPath.name)
        token = sdk2.Vault.data('NEWS', 'yt_token')
        request = Request(admin_url + api_path, headers={"Authorization": "OAuth {}".format(token)})
        r = urlopen(request)
        text = r.read()
        data = json.loads(text)

        annotations = []
        for annotation in data["annotations"]:
            annotations.append(TagMapping(annotation))
        annotations.sort(key=lambda annotation: (annotation.tag, annotation.component))

        output_path = self.ctx.get(Params.OutputFileName.name)
        with open(output_path, "w") as w:
            for annotation in annotations:
                w.write(annotation.to_line())
        signature = self._calculate_file_signature(output_path)

        resource = self.create_resource(
            description=self.descr,
            resource_path=output_path,
            resource_type=NEWS_THEMES_TAGS_TO_COMPONENTS,
            arch=self.arch,
            attributes={
                "signature": signature,
            },
        )

        self.mark_resource_ready(resource.id)

        need_release = True
        try:
            need_release = self._check_need_release(signature)
        except:
            logging.exception("Unable to check previous release")
            need_release = True
        return need_release

    @staticmethod
    def _calculate_file_signature(path):
        sha1_sign = hashlib.sha1()
        with open(str(path), "rb") as f:
            for chunk in iter(lambda: f.read(4194304), b""):
                sha1_sign.update(chunk)
        return sha1_sign.hexdigest()

    def _check_need_release(self, new_resource_signature):
        prev_resource = self._get_prev_released_resource()
        if not prev_resource:
            return True
        logging.debug("Found old release: %s", prev_resource.id)
        prev_resource_path = self.sync_resource(prev_resource)  # also touch
        prev_resource_signature = self._calculate_file_signature(prev_resource_path)
        logging.debug("Hashes: previous: %s, current: %s", prev_resource_signature, new_resource_signature)
        if (prev_resource_signature != new_resource_signature) or prev_resource_signature is None or new_resource_signature is None:
            logging.debug("Different hashes")
            return True
        logging.info("Same resource; no need to re-release")
        self.set_info("Resource is equal to the last release: {}; will skip auto release".format(link_builder.sb_item_link(prev_resource.id, "resource")), do_escape=False)
        return False

    def _get_prev_released_resource(self):
        try:
            resource = apihelpers.get_last_resource_with_attrs(NEWS_THEMES_TAGS_TO_COMPONENTS, all_attrs=True, attrs={'released': 'stable'})
            return resource
        except:
            logging.exception("Unable to find previous version of resource")
        return None


__Task__ = DownloadNewsThemesTagsToComponents
