import os.path
import logging
from yaml import load as yaml_load

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.juggler import jclient
from sandbox.sdk2.vcs.svn import Arcadia


class FindNewsUndeclaredFlags(sdk2.Task):
    """ This task looks for flags in directory yweb/news, which are NOT in persistent_flags.yaml """

    def on_execute(self):
        logging.info(Arcadia.trunk_url('yweb/news/'))
        Arcadia.checkout(Arcadia.trunk_url('yweb/news/'), os.path.abspath('tmp'))

        raw_flags = sp.check_output(
            "grep -onraP 'yxnews_[A-Za-z0-9_-]*[A-Za-z0-9](?![A-Za-z0-9_])' tmp", shell=True).splitlines()
        logging.info("Total flags count: " + str(len(raw_flags)))

        with open('tmp/services/app_host/routerd/persistent_flags.yaml', 'r') as f:
            pf_raw = yaml_load(f)

        pf = []
        for section_name, flags in pf_raw["flags"].items():
            pf += flags.keys()
        logging.info("Persistent flags count: " + str(len(pf)))

        flags = {}
        for line in raw_flags:
            parsed_line = line.decode('utf8').split(':')
            if len(parsed_line) != 3:
                logging.info("Here is bad line: " + line)
                continue
            if parsed_line[0] == 'tmp/services/app_host/routerd/persistent_flags.yaml':
                continue
            if parsed_line[2] in pf:
                continue
            if parsed_line[0].startswith("tmp/.svn"):
                continue
            logging.info("UNDECLARED FLAG BEGIN")
            logging.info("Filename: " + parsed_line[0])
            logging.info("Line: " + parsed_line[1])
            logging.info("Flag: " + parsed_line[2])
            blame = Arcadia.blame(parsed_line[0]).splitlines()[int(parsed_line[1]) - 1].decode('utf8').split()[1]
            logging.info("Blame: " + blame)
            logging.info("UNDECLARED FLAG END")
            if parsed_line[2] not in flags:
                flags[parsed_line[2]] = {blame}
            else:
                flags[parsed_line[2]].add(blame)
        logging.info("Undeclared flags count: " + str(len(flags)))

        if len(flags) <= 2:
            juggler_status = 'OK'
        elif len(flags) <= 10:
            juggler_status = 'WARN'
        else:
            juggler_status = 'CRIT'

        output_info = []
        for flag, blame in flags.items():
            output_info.append(flag + ' (' + ', '.join(blame) + ')')
        jclient.send_events_to_juggler(
            'find_news_undeclared_flags',
            'news_undeclared_flags_alert',
            juggler_status,
            'Undeclared flags: ' + ', '.join(output_info),
        )
