import logging

from sandbox import sdk2
import sandbox.sdk2.parameters as sp

import startrek_client
from startrek_client import Startrek
import datetime
import dateutil
import dateutil.tz
import dateutil.parser
from functools import reduce
from time import sleep


zone = dateutil.tz.gettz('Europe/Moscow')

comment_template = '''кто:{author}
{text}'''

goal_template = '''{parents}
==={goal_link}
В главных ролях: {people}
**Критерии**:
{criteria}

**Статус**:
{comments}
'''

report_template = '''Отчетный период: {since} - {to}

{goals}'''


def deep_get(dictionary, *keys):
    return reduce(lambda d, key: d.get(key) if d else None, keys, dictionary)


def fixtime(t):
    return datetime.datetime.fromtimestamp(t.timestamp())


def parse_date(s):
    d = datetime.datetime.strptime(s, '%Y%m%d')
    return fixtime(d.replace(tzinfo=zone))


def shiftlines(s, amount):
    pad = '  ' * amount
    return "\n".join(pad + x for x in s.splitlines())


def format_comments(comments):
    if comments:
        return '\n\n'.join(comment_template.format(author=c.createdBy.id, text=c.text) for c in comments)
    else:
        return '!!**НЕТ СТАТУСА**!!'


def format_deadline(d):
    date = dateutil.parser.isoparse(d['date']).date()
    now = datetime.datetime.now().date().replace(day=1)
    tp = d['deadlineType']
    if tp == 'quarter':
        now = now.replace(month=(now.month - 1) // 3 + 1)
        expired = now > date
        return ('Q{q} {y}'.format(q=(date.month - 1) // 3 + 1,
                                  y=date.year),
                expired)
    elif tp == 'date':
        now = now.replace(day=1)
        expired = now > date
        return (date.strftime('%b %Y'),
                expired)
    else:
        raise RuntimeError('Unknown deadline type {}'.format(tp))


def format_criterion(c):
    dl = c.get('deadline')
    (dlstr, expired) = format_deadline(dl) if dl else ('!!Без дедлайна!!', None)
    checked = c['checked']
    color = 'red' if expired and not checked else 'green'
    text = c['text']
    msg = '--{}--'.format(text) if checked else text
    cmsg = text if expired is None else '!!({color}){text}!!'.format(color=color, text=msg)
    return '{cmsg} {date}'.format(cmsg=cmsg,
                                  date=dlstr)


def goal_link(key):
    queue, id = key.split('-')
    assert(queue.lower() == 'goalz')
    return 'https://goals.yandex-team.ru/filter?goal={goal}'.format(goal=id)


def format_goal(i, parents, comments):
    pstr = "\n↳".join(goal_link(x.key) for x in parents)
    glink = ('↳' if parents else '') + goal_link(i.key)

    return goal_template.format(goal_link=glink,
                                parents=pstr,
                                people=', '.join('кто:' + u.login for u in i.participants),
                                criteria="\n".join('* {}'.format(format_criterion(c)) for c in i.checklist_items
                                                   if c['checklistItemType'] == 'criterion'),
                                comments=format_comments(comments))


class Reporter(object):
    def __init__(self, client, t0, t1, ignore_root=False):
        self.t = client
        self.t0 = t0
        self.t1 = t1
        self.ignore_root = ignore_root

        self.tocall = {}

    def _gen_report_for_goal(self, i, parents):
        try:
            subtasks = [x.object for x in i.links if x.direction == 'outward' and x.type.id == 'subtask']
            def goodcom(c):
                t = fixtime(dateutil.parser.isoparse(c.createdAt))
                return t >= self.t0 and t < self.t1

            if len(parents) or not self.ignore_root:
                comments = [x for x in i.comments if goodcom(x)]
                if not len(comments):
                    u = i.assignee
                    if u:
                        self.tocall.setdefault(u.login, []).append(i)
                selfrep = format_goal(i, parents[1:], comments) + '\n'
            else:
                selfrep = ''

            return selfrep + '\n'.join(
                self._gen_report_for_goal(st, parents + [i]) for st in subtasks)
        except startrek_client.exceptions.Forbidden:
            logging.warning('Forbidden to access task {}'.format(i.key))
            return '!!**Нет доступа к {}**!!'.format(i.key)

    def gen_report_for_goal(self, key):
        issue = self.t.issues[key]
        return report_template.format(since=self.t0.strftime('%Y-%m-%d'),
                                      to=self.t1.strftime('%Y-%m-%d'),
                                      goals=self._gen_report_for_goal(issue, []))


def call_to_comments(issue, tocall, comment=None):
    def userblock(u, iss):
        return u + "@:\n" + "\n".join(goal_link(i.key) for i in iss)

    text = "У нас есть цели без актуального статуса. Пожалуйста, найди себя ниже, напиши в каждой из своих целей краткий статус и актуализируй критерии\n\n" + "\n\n".join(
        userblock(u, iss) for (u, iss) in tocall.items())

    action = comment.update if comment else issue.comments.create
    action(text=text, summonees=list(tocall.keys()))


class GenNewsGoalsReport(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        st_token = sp.YavSecret('Tracker token', required=True)
        start_date = sp.String('Period start date, yyyymmdd')
        period_duration = sp.Integer('Period duration, days', required=True)
        report_date = sp.String('Report date, yyyymmdd. Empty = now')
        goal = sp.String('Root task name (GOALZ-XXX)', required=True)
        skip_root = sp.Bool("Don't try to generate status for root task", default=True)
        queue = sp.String("ST queue to create ticket in. Empty = don't create")
        clear_links = sp.Bool('Remove links of created task', default=True)
        retry_attempts = sp.Integer('Call responsibles of the goals without status and re-generate status N times', default=0)
        retry_delay = sp.Integer('Delay between regeneration, minutes', default=60)

    class Context(sdk2.Context):
        issue = None

    def on_execute(self):
        p = self.Parameters
        ctx = self.Context
        token, *_ = p.st_token.data().values()
        st = Startrek('news', token=token)
        duration = datetime.timedelta(days=p.period_duration)
        rep = parse_date(p.report_date) if p.report_date else datetime.datetime.now()
        start = parse_date(p.start_date) if p.start_date else datetime.datetime.now() - duration

        n = int((rep - start) / duration)
        beg = start + (n * duration)
        logging.info('Beg = {}, rep = {}'.format(beg, rep))

        r = Reporter(st, beg, rep, ignore_root=p.skip_root)
        report = r.gen_report_for_goal(p.goal)
        logging.info(report)

        if (p.queue):
            with self.memoize_stage.create_issue:
                task = st.issues.create(queue=p.queue,
                                        summary='Статус {}'.format(rep.strftime('%Y-%m-%d')),
                                        description=report)
                url = 'https://st.yandex-team.ru/{}'.format(task.key)
                self.set_info(info='Created task: <a href="{url}">{key}</a>'.format(url=url, key=task.key),
                              do_escape=False)
                if p.clear_links:
                    logging.info('Waiting for links to settle')
                    sleep(15)
                    logging.info('Clearing links')
                    [l.delete() for l in task.links]
                ctx.issue = task.key

            if p.retry_attempts:
                with self.memoize_stage.call_users(max_runs=p.retry_attempts + 1):
                    issue = st.issues[ctx.issue]
                    issue.update(description=report)
                    myself = st.myself.login
                    comment = next((c for c in issue.comments if c.createdBy.login == myself), None)
                    if r.tocall:
                        call_to_comments(issue, r.tocall, comment=comment)
                        raise sdk2.WaitTime(60 * p.retry_delay)
        else:
            self.set_info("Report:\n{}".format(report))
