# -*- coding: utf-8 -*-

import json
import os
import six

from sandbox import sdk2
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.news.arranged import Arranged, ArrangedParameters
from sandbox.projects.common import error_handlers as eh

from sandbox.projects.news import resources


class GetArrangedApphostResponses(sdk2.Task):
    '''
        Получает и сохраняет результаты обстрела arranged apphost'овыми запросами
    '''

    class Parameters(ArrangedParameters):
        app_host_queries = sdk2.parameters.Resource('arranged apphost requests', resource_type=resources.ARRANGED_APPHOST_REQUESTS, required=True)
        requester = sdk2.parameters.Resource('app host requester', resource_type=resource_types.SLAVE_NEWSD_APPHOST_REQUESTER, required=False)
        fail_on_empty_answer = sdk2.parameters.Bool('Fail if got empty answer', default=False)

    client_tags = ctc.Tag.Group.LINUX

    def on_enqueue(self):
        self.Context.out_resource_id = resources.ARRANGED_APPHOST_RESPONSES(self, "arranged apphost responses", "arranged_apphost_responses.txt").id

    def on_execute(self):
        out_resource = sdk2.ResourceData(sdk2.Resource[self.Context.out_resource_id])

        workdir = str(self.path('work'))
        os.makedirs(workdir)
        port = 17190

        attrs = {"released": "stable"}
        # RECOMMENDER SYSTEM CONFIG
        if self.Parameters.recommender_config is not None:
            recommender_config = str(sdk2.ResourceData(self.Parameters.recommender_config).path)
        else:
            resource = sdk2.Resource.find(
                resources.NEWS_RECOMMENDER_SYSTEM_CONFIG,
                attrs=attrs).order(-sdk2.Resource.id).first()
            recommender_config = str(sdk2.ResourceData(resource).path)

        # DEFAULT FLAGS
        if self.Parameters.default_flags is not None:
            default_flags_config = str(sdk2.ResourceData(self.Parameters.default_flags).path)
        else:
            resource = sdk2.Resource.find(
                resources.NEWS_APPHOST_ARRANGED_DEFAULT_FLAGS,
                attrs=attrs).order(-sdk2.Resource.id).first()
            default_flags_config = str(sdk2.ResourceData(resource).path)

        # RECOMMENDER MODELS
        if self.Parameters.recommender_models is not None:
            recommender_models = str(sdk2.ResourceData(self.Parameters.recommender_models).path)
        else:
            resource = sdk2.Resource.find(
                resources.NEWS_RECOMMENDER_MODELS_PACKAGE,
                attrs=attrs).order(-sdk2.Resource.id).first()
            recommender_models = str(sdk2.ResourceData(resource).path)

        arranged = Arranged(binary=str(sdk2.ResourceData(self.Parameters.arranged_executable).path),
                            port=port,
                            recommender_config=recommender_config,
                            default_flags_config=default_flags_config,
                            workdir=workdir,
                            recommender_models=recommender_models,
                            state=str(sdk2.ResourceData(self.Parameters.state).path),
                            )

        requester = None
        if self.Parameters.requester is not None:
            requester = str(sdk2.ResourceData(self.Parameters.requester).path)
        else:
            resource = sdk2.Resource.find(
                resource_types.SLAVE_NEWSD_APPHOST_REQUESTER).order(-sdk2.Resource.id).first()
            requester = str(sdk2.ResourceData(resource).path)

        requests = str(sdk2.ResourceData(self.Parameters.app_host_queries).path)

        timestamp = channel.sandbox.get_resource_attribute(
            self.Parameters.app_host_queries.id,
            attribute_name="timestamp",
        )
        arranged.start(True, timestamp)
        arranged.wait()

        cmd = [
            requester,
            '--address', "post://localhost:{}".format(str(arranged.get_apphost_port())),
            '--input', requests,
            '--output', str(out_resource.path),
        ]
        self.process_parameters = cmd

        self.process = process.run_process(cmd, log_prefix='requester_', wait=True)

        out_resource.ready()

        self._validate_answers(str(out_resource.path))

    def _validate_answers(self, file_name):
        def iter_responses():
            with open(file_name) as fd:
                for line in fd:
                    line = line.rstrip('\r\n')
                    if line:
                        fields = line.split('\t')
                        response_str = fields[2]
                        response_str = six.text_type(response_str, errors='replace')
                        yield response_str
        total, empty = 0, 0
        for response_str in iter_responses():
            total += 1
            r = json.loads(response_str)
            if not r or not r.get("answers", []):
                empty += 1
        if empty:
            (eh.check_failed if self.Parameters.fail_on_empty_answer else self.set_info)("{} out of {} answers are empty".format(empty, total))
