# -*- coding: utf-8 -*-

import json
import os
import sys
import requests

from sandbox import sdk2
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.news.newsd import SlaveNewsd
from sandbox.projects.common import network
from sandbox.projects.common import error_handlers as eh

from sandbox.projects.news import resources
from sandbox.projects.geobase.GeodataTreeLingStable.resource import GEODATA_TREE_LING_STABLE


class GetHuaweiApiResponses(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        slave_newsd_executable = sdk2.parameters.Resource(
            'slave_newsd executable', resource_type=resource_types.SLAVE_NEWSD_EXECUTABLE, required=True
        )
        slave_newsd_config = sdk2.parameters.Resource(
            'slave_newsd config', resource_type=resource_types.SLAVE_NEWSD_CONFIG, required=True
        )
        slave_newsd_state = sdk2.parameters.Resource(
            'slave_newsd state dump', resource_type=resource_types.SLAVE_NEWSD_STATE, required=True
        )
        slave_newsd_geodata = sdk2.parameters.Resource(
            'slave_newsd geodata', resource_type=GEODATA_TREE_LING_STABLE, required=True
        )
        slave_newsd_index_config = sdk2.parameters.Resource(
            'slave_newsd index config', resource_type=resources.NEWS_DEFAULT_INDEX_CONFIG, required=True
        )
        api_responder_executable = sdk2.parameters.Resource(
            'api_responder executable', resource_type=resources.NEWS_APPHOST_API_RESPONDER, required=True
        )
        api_queries = sdk2.parameters.Resource('huawei api requests', resource_type=resources.NEWS_HUAWEI_API_REQUESTS, required=True)

    client_tags = ctc.Tag.Group.LINUX

    def on_enqueue(self):
        self.Context.out_resource_id = resources.NEWS_HUAWEI_API_RESPONSES(self, "huawei api responses", "huawei_api_responses.txt").id

    def start_api_responder(self, port, workdir):
        api_responder_log = workdir + '/api_responder.log'
        # api_responder_log = paths.get_unique_file_name(paths.get_logs_filter(), "api_repsonder.log")
        cmd = [
            str(sdk2.ResourceData(self.Parameters.api_responder_executable).path),
            '--port', str(port),
            '--log_path', api_responder_log,
        ]
        self.process_parameters = cmd
        self.process = process.run_process(cmd, log_prefix='api_responder', wait=False)

    def on_execute(self):
        out_resource = sdk2.ResourceData(sdk2.Resource[self.Context.out_resource_id])

        workdir = str(self.path('work'))
        os.makedirs(workdir)
        newsd_port = 17190
        api_responder_port = 17180

        timestamp = channel.sandbox.get_resource_attribute(
            self.Parameters.slave_newsd_state.id,
            attribute_name="timestamp",
        )

        slave_newsd = SlaveNewsd(
            binary=str(sdk2.ResourceData(self.Parameters.slave_newsd_executable).path),
            cfg=str(sdk2.ResourceData(self.Parameters.slave_newsd_config).path),
            port=newsd_port,
            geobase=str(sdk2.ResourceData(self.Parameters.slave_newsd_geodata).path),
            index_config_path=str(sdk2.ResourceData(self.Parameters.slave_newsd_index_config).path),
            state=str(sdk2.ResourceData(self.Parameters.slave_newsd_state).path),
            workdir=workdir,
            app_host_mode=True,
            now=timestamp,
        )
        self.start_api_responder(api_responder_port, workdir)
        slave_newsd.start()
        slave_newsd.wait()

        requests_path = str(sdk2.ResourceData(self.Parameters.api_queries).path)
        host = network.get_my_ipv6()

        with open(requests_path, 'r') as requests_file:
            with out_resource.path.open('w') as out_file:
                for line in requests_file:
                    requestJson = json.loads(line)
                    path = requestJson['path']
                    hasParams = False
                    if 'params' in requestJson:
                        params = requestJson['params']
                        if params:
                            hasParams = True
                            path += '?'
                            for param in params:
                                path += param + '=' + params[param] + '&'
                    path = 'http://news.stable.priemka.yandex.ru' + path
                    if not hasParams:
                        path += '?'
                    path += 'srcrwr=NEWS_API_RESPONDER:[{}]:{}&srcrwr=NEWS_SLAVE_NEWSD:[{}]:{}'.format(host, api_responder_port, host, (newsd_port + 2))
                    payload = None
                    if 'body' in requestJson:
                        payload = requestJson['body']

                    response = requests.post(path, data=payload)
                    try:
                        # response_json = response.json()
                        out_file.write(unicode(response.text, 'utf-8'))
                    except Exception as e:
                        sys.stderr.write(str(response.status_code))
                        sys.stderr.write('\n')
                        sys.stderr.write(str(response.reason))
                        sys.stderr.write('\n')
                        sys.stderr.write(str(response.text))
                        sys.stderr.write('\n')
                        eh.check_failed("Error for " + path + " with error: " + str(e))
