# coding: utf-8

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia as arc
from sandbox.sandboxsdk import ssh

import sandbox.common.types.task as ctt

import logging

USER = 'robot-ynews-ro'


def svn_list_numeric(url):
    def fix(entry):
        if entry[-1] == '/':
            entry = entry[:-1]
        return entry

    return sorted([int(fix(x)) for x in arc.list(url, as_list=True)])


class MakeNewsRelease(sdk2.Task):
    """
    Make news release branch/tag and build it
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        environments = (environments.SvnEnvironment(),)
        disk_space = 4096
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 600
        max_restarts = 0

        with sdk2.parameters.String("Create new:", multiline=True) as action:
            action.values.BRANCH = "Branch and tag"
            action.values.TAG = "Tag"

        branch = sdk2.parameters.Integer("Branch for new tag (0 = last)", default=0)
        build_newsd = sdk2.parameters.Bool("Build slave_newsd", default=True)
        build_indexer = sdk2.parameters.Bool("Build indexer package", default=True)

    BRANCHES = "arcadia:/arc/branches/news/releases/indexer"
    TAGS = 'arcadia:/arc/tags/news/releases/indexer'
    TRUNK = 'arcadia:/arc/trunk'

    def svncopy(self, *args, **kwargs):
        kwargs['user'] = USER
        with ssh.Key(self, key_owner='NEWS', key_name='robot-ynews-ro-key'):
            return arc.copy(*args, **kwargs)

    def branch_url(self, branch):
        return '{}/{}'.format(self.BRANCHES, branch)

    def tag_url(self, branch, tag):
        return '{}/{}/{}'.format(self.TAGS, branch, tag)

    def get_last_branch(self):
        return svn_list_numeric(self.BRANCHES)[-1]

    def get_last_tag(self, branch):
        tags = svn_list_numeric('{}/{}'.format(self.TAGS, branch))
        return tags[-1] if len(tags) else 0

    def on_execute(self):
        with self.memoize_stage.branchtag:
            branch = self.Parameters.branch or self.get_last_branch()
            if self.Parameters.action == 'BRANCH':
                branch += 1
                burl = self.branch_url(branch)
                self.svncopy(self.TRUNK, burl, 'Create branch {}'.format(branch))
                logging.info('Created {}'.format(burl))

            tag = self.get_last_tag(branch) + 1

            turl = self.tag_url(branch, tag)
            self.svncopy(self.branch_url(branch), turl, 'Create tag {}/{}'.format(branch, tag), parents=True)

            logging.info('Created {}'.format(turl))

            relname = 'release-{}.{}'.format(branch, tag)

            self.Parameters.description = '{} - {}'.format(self.Parameters.description, relname)

            self.Context.tagname = '{}/{}'.format(branch, tag)

            common_params = {
                'build_type': 'release',
                'checkout_mode': 'manual',
                'checkout': True,
                'checkout_arcadia_from_url': turl + '/arcadia',
                'clear_build': True,
                'check_return_code': True,
            }

            subtasks = []

            if self.Parameters.build_indexer:
                pkg_task = sdk2.Task['BUILD_NEWS_PACKAGE'](self, description=relname,
                                                           custom_bins_resource_id=93312847,
                                                           basesearch_resource_id=195206405,
                                                           do_build_news='build',
                                                           do_build_news_data='build',
                                                           do_build_news_scripts='build',
                                                           do_build_basesearch='',
                                                           news_data_checkout_arcadia_from_url=turl + '/arcadia_tests_data',
                                                           perl_modules_resource_id=407504142,
                                                           **common_params)
                pkg_task.enqueue()
                subtasks.append(pkg_task)

            if self.Parameters.build_newsd:
                newsd_task = sdk2.Task['BUILD_SEARCH'](self, description='slave_newsd, r{}.{}'.format(branch, tag),
                                                       build_slave_newsd=True,
                                                       **common_params)
                newsd_task.enqueue()
                subtasks.append(newsd_task)

            if len(subtasks):
                raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH, wait_all=True)

    @property
    def short_task_result(self):
        return 'release-{}'.format(self.Context.tagname)

    def get_short_task_result(self):
        return self.short_task_result
