# coding: utf-8
from sandbox import sdk2
from sandbox.sandboxsdk.task import SandboxTask as OldTask  # Ungraceful method of autorelease
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia as arc
from sandbox.sandboxsdk import ssh

import sandbox.common.types.task as ctt

import requests
from datetime import datetime
import json
import logging


USER = 'robot-ynews-ro'


def svn_list_numeric(url):
    def fix(entry):
        if entry[-1] == '/':
            entry = entry[:-1]
        return entry

    return sorted([int(fix(x)) for x in arc.list(url, as_list=True)])


def is_number(s):
    try:
        int(s)
        return True
    except ValueError:
        return False


def concat_entry(entry):  # Accepts commit message and revision from svn log; returns string
    # Example: "REVIEW: 1234\n"
    revisionlink = "((https://a.yandex-team.ru/arc/commit/" + str(entry["revision"]) + " [r" + str(entry["revision"]) + "]))"

    # Commiter may differ from ticket executor, so better be excessive than incorrect
    text = revisionlink + " " + str(entry["author"]) + "@ " + entry["msg"].strip()

    # Attach a url to review
    parsetext = text.lower()  # Sorry for not using Damerau–Levenshtein distance

    begin = parsetext.find("review")
    while begin != -1:
        blocks = parsetext[begin:].split("\n")[0].split(":")
        if len(blocks) >= 2 and blocks[0].strip() == "review" and is_number(blocks[1]):
            end = begin + len(blocks[0]) + 1 + len(blocks[1].rstrip())
            reviewlink = "((https://a.yandex-team.ru/review/" + blocks[1].strip() + " " + text[begin:end] + "))"
            text = text[:begin] + reviewlink + text[end:]

        begin = parsetext.find("review", begin + 1)

    return text


def merge_commit(entry):  # Returns list of numbers of commits to trunk. Made to delete these commits from changelog
    # Example: "merge from trunk: 123, 531, 654\n"
    commit_list = []

    text = entry["msg"].lower()
    begin = text.find("merge from trunk")

    while begin != -1:
        blocks = text[begin:].split("\n")[0].split(":")
        if len(blocks) >= 2 and blocks[0].strip() == "merge from trunk":
            for commit in blocks[1].split(","):
                if is_number(commit):
                    commit_list.append(int(commit))
                else:
                    break
            if len(commit_list) > 0:
                logging.info("OK: " + "Added merge commits - " + str(commit_list))

        begin = text.find("merge from trunk", begin + 1)

    return commit_list


class MakeNewsRelease2(sdk2.Task):
    """
    Make news release branch/tag and build it
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        environments = (environments.SvnEnvironment(),)
        disk_space = 4096
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 600
        max_restarts = 0

        with sdk2.parameters.String("Create new:", multiline=True) as action:
            action.values.DEBUG = "Save changelog locally"
            action.values.LOG = "Send changelog to Startrek"
            action.values.BRANCH = "Branch and tag"
            action.values.TAG = "Tag"

        branch = sdk2.parameters.Integer("Branch for new tag (0 = last branch)", default=0)
        build_newsd = sdk2.parameters.Bool("Build slave_newsd", default=True)
        build_indexer = sdk2.parameters.Bool("Build indexer package", default=True)
        build_yt_indexer = sdk2.parameters.Bool("Build indexer yt package", default=True)
        build_ranking_models = sdk2.parameters.Bool("Build ranking models package", default=True)
        build_slaveconf = sdk2.parameters.Bool("Build slave_newsd config", default=True)
        legacy_build = sdk2.parameters.Bool("Use legacy BUILD_NEWS_PACKAGE task", default=False)
        prevrelease = sdk2.parameters.Integer("Changelog is generated from branch A to B. Branch A (0 = last branch):", default=0)
        prevtag = sdk2.parameters.Integer("Tag of branch A (0 = don't use tag)", default=0)
        currelease = sdk2.parameters.Integer("Branch B (0 = current branch)", default=0)

        with sdk2.parameters.String("Release to", default='NONE') as release_subtasks_to:
            release_subtasks_to.values.NONE = 'Do not release'
            for n in ctt.ReleaseStatus.__names__:
                val = getattr(ctt.ReleaseStatus, n)
                setattr(release_subtasks_to.values, val, val)

    BRANCHES_PATH = "/arc/branches/news/releases/indexer"
    TAGS_PATH = "/arc/tags/news/releases/indexer"
    TRUNK_PATH = "/arc/trunk"
    NEWS_PATH = "/arc/trunk/arcadia/yweb/news"

    BRANCHES = "arcadia:" + BRANCHES_PATH
    TAGS = "arcadia:" + TAGS_PATH
    TRUNK = "arcadia:" + TRUNK_PATH
    NEWS = "arcadia:" + NEWS_PATH

    INDEXER_NAME = "BUILD_NEWS_PACKAGE"
    INDEXER_YT_NAME = "BUILD_NEWS_INDEXER_YT_PACKAGE"
    RANKING_MODELS_NAME = "BUILD_NEWS_RANKING_MODELS_PACKAGE"
    SLAVE_NAME = "BUILD_SEARCH"
    SLAVE_CONFIG = "/arcadia/yweb/news/services/slave_newsd/slave_newsd_data.conf"
    CONFIG_NAME = "BUILD_NEWS_SLAVE_NEWSD_SERVICE_CONFIG_BUNDLE"
    SLAVE_SERVICE_CONFIGS = "/arcadia/yweb/news/services/slave_newsd"
    SLAVE_CONFIGS_ADDITIONAL = [
        '/arcadia/yweb/news/config/default_index_config'
    ]

    def svncopy(self, *args, **kwargs):
        kwargs['user'] = USER
        with ssh.Key(self, key_owner='NEWS', key_name='robot-ynews-ro-key'):
            return arc.copy(*args, **kwargs)

    def svnlog(self, *args, **kwargs):
        with ssh.Key(self, key_owner='NEWS', key_name='robot-ynews-ro-key'):
            return arc.log(*args, **kwargs)

    def branch_url(self, branch):
        return '{}/{}'.format(self.BRANCHES, branch)

    def tag_url(self, branch, tag):
        return '{}/{}/{}'.format(self.TAGS, branch, tag)

    def get_last_branch(self):  # Returns int
        return svn_list_numeric(self.BRANCHES)[-1]

    def get_last_tag(self, branch):  # Returns int
        tags = svn_list_numeric('{}/{}'.format(self.TAGS, branch))
        return tags[-1] if len(tags) else 0

    # *** Changelog - Startrek ***
    def to_revision(self, url):  # Returns revision (string) at which url was created
        return str(self.svnlog(url, None, stop_on_copy=True)[-1]["revision"])

    def branch_rev(self, branch):  # Returns revision (string) at which branch was created
        return self.to_revision(self.branch_url(branch))

    def tag_rev(self, branch, tag):  # Returns revision (string) at which tag of branch was created
        return self.to_revision(self.tag_url(branch, tag))

    def slice_of_history(self, index1, tag1, index2):  # Returns formatted changelog text from release <index1.tag1> to release <index2>
        begin_rev = ""
        if tag1 == 0:
            begin_rev = self.branch_rev(index1)
        else:
            begin_rev = self.tag_rev(index1, tag1)
        res = "".join(["Changes in ", self.NEWS_PATH,
                       " from release ", str(index1), ("." + str(tag1)) * (tag1 != 0),
                       " to ", str(index2),
                       " (revision r", begin_rev, ":r", self.branch_rev(index2), ")\n",
                       "Sandbox task ((https://sandbox.yandex-team.ru/task/", str(self.id), "/view #", str(self.id), "))\n\n"])

        # We do not show commits to branches because they already are in production
        merges = set()
        commits_list = self.svnlog(self.BRANCHES, self.branch_rev(index1), self.branch_rev(index2))
        for entry in commits_list:
            merges |= set(merge_commit(entry))

        for rel in range(index2, index1, -1):
            res += " " * 8 + "**Release " + str(rel) + "**\n"

            a = self.branch_rev(rel - 1)  # Starting revision. Tag != 0 -> use tag's revision instead of branch's
            if rel == index1 + 1 and tag1 != 0:
                a = begin_rev
            b = self.branch_rev(rel)
            changes = self.svnlog(self.NEWS, a, revision_to=b)
            for entry in changes[::-1]:
                if not entry["revision"] in merges:
                    res += concat_entry(entry) + "\n"
                else:
                    logging.info("OK: " + "Removed commit " + str(entry['revision']) + " from changelog")
        return res

    # Startrek tickets. This task is executed by "robot-ynews", who must have access to NEWREL queue
    def create_ticket(self, summary, text):  # Returns Startrek's response to ticket creation request
        auth = {"Authorization": "OAuth " + sdk2.Vault.data("NEWS", "robot-startrek-key"), "Content-Type": "application/json"}
        request_params = {
            "queue": "NEWREL",
            "type": 2,
            "summary": summary,
            "description": text,
        }
        req = json.dumps(request_params, ensure_ascii=False, indent=2)
        logging.debug("Startrek request: %s", req)
        r = requests.post("https://st-api.yandex-team.ru/v2/issues", headers=auth, data=req)
        return r.text.encode("utf-8")

    # Master function for posting changelog in startrek
    def create_changelog(self, prevbranch, prevtag, branch):
        logtext = self.slice_of_history(prevbranch, prevtag, branch).encode("utf-8")
        ticket_summary = "Release " + str(branch)
        if self.Parameters.action == "LOG":
            ticket_summary = "Changelog from release " + str(prevbranch) + ("." + str(prevtag)) * (prevtag != 0) + " to " + str(branch)
        response = self.create_ticket(ticket_summary, logtext)
        logging.info("Startrek responded:\n" + response)

    def on_execute(self):
        with self.memoize_stage.branchtag:
            branch = self.Parameters.branch or self.get_last_branch()
            rel_a = self.Parameters.prevrelease
            tag = self.Parameters.prevtag
            rel_b = self.Parameters.currelease

            # Changelog is not generated on TAG operation
            # Changelog text can be seen in DEBUG in Sandbox task logs
            # LOG only sends the ticket with changelog text and performs no writes in Arcadia
            if self.Parameters.action == "DEBUG":
                logging.info("HISTORY:\n" + self.slice_of_history(rel_a, tag, rel_b))
            elif self.Parameters.action == "LOG":
                rel_a = rel_a or (branch - 1)
                rel_b = rel_b or branch
                self.create_changelog(rel_a, tag, rel_b)
            else:
                if self.Parameters.action == 'BRANCH':
                    rel_a = rel_a or branch
                    rel_b = rel_b or (branch + 1)
                    branch += 1

                    burl = self.branch_url(branch)
                    self.svncopy(self.TRUNK, burl, 'Create branch {}'.format(branch))
                    logging.info('Created {}'.format(burl))

                    self.create_changelog(rel_a, tag, rel_b)  # Create changelog ticket only if branch is created

                tag = self.get_last_tag(branch) + 1
                turl = self.tag_url(branch, tag)
                self.svncopy(self.branch_url(branch), turl, 'Create tag {}/{}'.format(branch, tag), parents=True)
                logging.info('Created {}'.format(turl))

            # Launch child tasks
            relname = 'release-{}.{}'.format(branch, tag)
            self.Parameters.description = '{} - {}'.format(self.Parameters.description, relname)
            self.Context.tagname = '{}/{}'.format(branch, tag)

            common_params = {
                'build_type': 'release',
                'checkout_mode': 'auto',
                'checkout': True,
                'checkout_arcadia_from_url': self.tag_url(branch, tag) + '/arcadia',
                'clear_build': True,
                'check_return_code': True,
            }

            subtasks = []
            self.Context.subtasks = dict()

            if self.Parameters.build_indexer:
                if self.Parameters.legacy_build:
                    pkg_task = sdk2.Task[self.INDEXER_NAME](self, description=relname,
                                                            custom_bins_resource_id=93312847,
                                                            basesearch_resource_id=195206405,
                                                            do_build_news='build',
                                                            do_build_news_data='build',
                                                            do_build_news_scripts='build',
                                                            do_build_basesearch='',
                                                            use_aapi_fuse=True,
                                                            news_data_checkout_arcadia_from_url=turl + '/arcadia_tests_data',
                                                            perl_modules_resource_id=407504142,
                                                            **common_params)
                else:
                    pkg_task = sdk2.Task['BUILD_NEWS_INDEXER_PACKAGE'](self, description=relname,
                                                                       tags=["NEWS-INDEXER"],
                                                                       **common_params)
                pkg_task.enqueue()
                subtasks.append(pkg_task)
                self.Context.subtasks[self.INDEXER_NAME] = pkg_task.id

            if self.Parameters.build_yt_indexer:
                yt_pkg_task = sdk2.Task[self.INDEXER_YT_NAME](self, description=relname,
                                                              strip_binaries=True,
                                                              tags=["NEWS-INDEXER"],
                                                              **common_params)
                yt_pkg_task.enqueue()
                subtasks.append(yt_pkg_task)
                self.Context.subtasks[self.INDEXER_YT_NAME] = yt_pkg_task.id

            if self.Parameters.build_ranking_models:
                ranking_models_pkg_task = sdk2.Task[self.RANKING_MODELS_NAME](self,
                                                                              description=relname,
                                                                              tags=["NEWS-INDEXER"],
                                                                              **common_params)
                ranking_models_pkg_task.enqueue()
                subtasks.append(ranking_models_pkg_task)
                self.Context.subtasks[self.RANKING_MODELS_NAME] = ranking_models_pkg_task.id

            if self.Parameters.build_newsd:
                newsd_task = sdk2.Task[self.SLAVE_NAME](self, description='slave_newsd, r{}.{}'.format(branch, tag),
                                                        build_slave_newsd=True,
                                                        **common_params)
                newsd_task.enqueue()
                subtasks.append(newsd_task)
                self.Context.subtasks[self.SLAVE_NAME] = newsd_task.id

            if self.Parameters.build_slaveconf:
                slaveconf_task = sdk2.Task[self.CONFIG_NAME](self,
                                                             description='new config bundle ' + datetime.now().strftime('%d.%m.%Y'),
                                                             news_resources_arcadia_path=self.tag_url(branch, tag) + self.SLAVE_SERVICE_CONFIGS,
                                                             override_resource_paths={
                                                                 'default_index_config': self.tag_url(branch, tag) + self.SLAVE_CONFIGS_ADDITIONAL[0]
                                                             },
                                                             **common_params)
                slaveconf_task.enqueue()
                subtasks.append(slaveconf_task)
                self.Context.subtasks[self.CONFIG_NAME] = slaveconf_task.id

            if len(subtasks):
                raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH, wait_all=True)
        with self.memoize_stage.autorelease(commit_on_entrance=False):
            if self.Parameters.release_subtasks_to != 'NONE':
                logging.info('release to: {}'.format(self.Parameters.release_subtasks_to))
                for taskname in [self.SLAVE_NAME, self.CONFIG_NAME, self.INDEXER_NAME, self.RANKING_MODELS_NAME, self.INDEXER_YT_NAME]:
                    if taskname in self.Context.subtasks:
                        logging.info("Automatic release of " + taskname)
                        # Sandbox cannot serialize task objects => pass ids via self.Context
                        OldTask(self).create_release(self.Context.subtasks[taskname],
                                                     status=self.Parameters.release_subtasks_to,
                                                     subject="Automatic release {}".format(self.Context.tagname))

    @property
    def short_task_result(self):
        return 'release-{}'.format(self.Context.tagname)

    def get_short_task_result(self):
        return self.short_task_result
