from datetime import date
import calendar
import os

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.projects.common.juggler import jclient


class MakeNewsRuntimeRelease(sdk2.Task):

    environment = (
        environments.SvnEnvironment(),
    )

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1 * 1024
        ram = 1 * 1024

    class Parameters(sdk2.Task.Parameters):
        description = "Task for news runtime release"

        with sdk2.parameters.CheckGroup("Release days", required=True) as release_days:
            release_days.values.Monday = release_days.Value("Monday", checked=False)
            release_days.values.Tuesday = release_days.Value("Tuesday", checked=False)
            release_days.values.Wednesday = release_days.Value("Wednesday", checked=False)
            release_days.values.Thursday = release_days.Value("Thursday", checked=False)
            release_days.values.Friday = release_days.Value("Friday", checked=False)
            release_days.values.Saturday = release_days.Value("Saturday", checked=False)
            release_days.values.Sunday = release_days.Value("Sunday", checked=False)

    def on_execute(self):
        try:
            today = calendar.day_name[date.today().weekday()]

            for day in self.Parameters.release_days:
                if day == today:
                    ya = sdk2.svn.Arcadia.export('arcadia:/arc/trunk/arcadia/ya', 'ya')
                    self.create_release(ya)
                    break

            self.send_signal('OK')
        except Exception as e:
            self.send_signal('CRIT')
            raise e

    def send_signal(self, status):
        self.set_info("Push to juggler: " + status)
        try:
            jclient.send_events_to_juggler(
                "news-runtime-release-scheduler",
                "news-runtime-release-scheduler",
                status,
                "Launches: https://sandbox.yandex-team.ru/tasks?children=false&hidden=false&type=MAKE_NEWS_RUNTIME_RELEASE"
            )
        except Exception:
            self.set_info('Failed to send event to juggler')

    def create_release(self, ya):
        with ProcessLog(self, 'bash_magic.log') as pl:
            environments.SvnEnvironment().prepare()

            env = os.environ.copy()
            env["HOME"] = os.getcwd()

            sp.check_call(
                [
                    ya, "tool", "news", "releaser", "init",
                    "--sb-token", sdk2.Vault.data("robot_ynews_mon_sandbox_token"),
                    "--st-token", sdk2.Vault.data("robot_ynews_mon_st_token"),
                    "--nanny-token", sdk2.Vault.data("robot_ynews_mon_nanny_token")
                ],
                stdout=pl.stdout, stderr=pl.stderr, env=env
            )

            filename = os.path.join(os.getcwd(), "yes.please")
            with open(filename, 'w') as f:
                f.write('y\ny\ny\n')

            with open(filename, 'r') as f:
                sp.check_call(
                    [ya, "tool", "news", "releaser", "release", "prestable"],
                    stdout=pl.stdout, stderr=pl.stderr, env=env, stdin=f
                )

    def on_failure(self, prev_status):
        self.set_info("on_failure called")
        self.send_signal('CRIT')
