# -*- coding: utf-8 -*-

import datetime
import logging
import json

from os.path import join as pj
from sandbox import common
from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments
from sandbox.projects.rthub.resources import RthubAdvWidgetFullPackage
from sandbox.projects.news.DownloadAdvWidgetClientsTable import AdvWidgetClientsFile


class LastResource(sdk2.parameters.Resource):

    resource_type = RthubAdvWidgetFullPackage.name
    required = True

    @common.utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            attrs={"released": ctt.ReleaseStatus.STABLE},
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class ProcessArchiveAdvwidgetTurboJsons(sdk2.Task):
    """
        Load items from turbo archive, annotate them and send to logbroker topic
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        environments = [environments.PipEnvironment("yandex-yt")]
        cores = 2
        disk_space = 50 * 1024

    class Parameters(sdk2.Parameters):
        description = "Load items from turbo archive, annotate them and send to logbroker topic"
        kill_timeout = 10 * 3600
        max_restarts = 3

        with sdk2.parameters.Group("Input parameters") as in_block:
            yt_cluster_name = sdk2.parameters.String("YT Cluster name", default='hahn', required=True)
            yt_path = sdk2.parameters.String("YT working dir path", required=False)
            yt_token = sdk2.parameters.Vault("Vault secret name with YT token",
                                             default='NEWS:yt_token',
                                             required=True)
            tvm_token = sdk2.parameters.Vault("Vault secret name with TVM secret",
                                              default='NEWS:tvm_secret',
                                              required=True)
            rthub_package = LastResource("Rthub advwidget package")
            archive_clients_list = sdk2.parameters.Resource("Rthub advwidget archive clients list with start end timestamps"
                                                            "(default is from [vanga].//home/news/advwidget/clients_list)",
                                                            resource_type=AdvWidgetClientsFile,
                                                            required=False,
                                                            multiple=False)
            hours = sdk2.parameters.Integer("Default hours to process", default=30 * 24, required=True)
            last_timestamp = sdk2.parameters.Integer("Default last timestamp to process", required=True,
                                                     default=int(datetime.datetime.now().strftime('%s')))
            default_time_segment = sdk2.parameters.Bool("Ignore time segment from list with clients", default=False)
            script_binary = sdk2.parameters.Resource('Script binary '
                                                     '(default is yweb/news/scripts/advwidget/yt_run_archive from package)',
                                                     required=False, multiple=False)

        with sdk2.parameters.Group("Output parameters") as out_block:
            write_to_stable = sdk2.parameters.Bool("Write to stable saas", default=False)
            write_to_prestable = sdk2.parameters.Bool("Write to prestable saas", default=True)
            write_archive = sdk2.parameters.Bool("Write new processed hosts to archive table", default=False)

    def read_table(self, proxy, yt_token, path):
        import yt.wrapper
        client = yt.wrapper.YtClient(proxy=proxy, token=yt_token)
        return set([json.dumps(json.loads(js), sort_keys=True) for js in client.read_table(path, format='json', raw=True)]) if client.exists(path) else set()

    def read_rsy_tables(self, yt_token):
        try:
            return self.read_table('arnold', yt_token, '//home/partner/domains_for_news')
        except:
            return self.read_table('hahn', yt_token, '//home/partner/domains_for_news')

    def clients_list(self, yt_token):
        if self.Parameters.archive_clients_list:
            return str(sdk2.ResourceData(self.Parameters.archive_clients_list).path)

        new_items = self.read_rsy_tables(yt_token)

        arch_table = '//home/news/advwidget/loaded_archive_clients_list'
        prev_items = self.read_table('vanga', yt_token, arch_table)
        new_items = new_items - prev_items
        logging.info(new_items)
        if len(new_items) > 0:
            with open('items.json', 'w') as f:
                f.write('\n'.join(str(line) for line in new_items))
            return 'items.json'
        return None

    def write_archive_table(self, yt_token):
        import yt.wrapper

        if self.Parameters.archive_clients_list:
            return str(sdk2.ResourceData(self.Parameters.archive_clients_list).path)
        client = yt.wrapper.YtClient(proxy='vanga', token=yt_token)
        tp = yt.wrapper.TablePath('//home/news/advwidget/loaded_archive_clients_list', append=True)
        client.write_table(tp, open('items.json'), format="json", raw=True)

    def on_execute(self):
        import yt.wrapper

        yt_token = self.Parameters.yt_token.data()
        clients_list = self.clients_list(yt_token)

        if clients_list is None:
            self.Parameters.description = 'No need to download archives, nothing has changed'
            return

        client = yt.wrapper.YtClient(proxy=self.Parameters.yt_cluster_name, token=yt_token)

        if self.Parameters.yt_path:
            yt_path = self.Parameters.yt_path
        else:
            yt_path = '//home/news/advwidget/' + str(self.id)
            _time = datetime.datetime.utcnow() + datetime.timedelta(hours=12)
            client.create("map_node", yt_path, recursive=True, attributes={"expiration_time": _time.isoformat()})
        yt_path = yt_path + '/'
        logging.info('Rthub yt workdir: ' + yt_path)

        package = str(sdk2.ResourceData(self.Parameters.rthub_package).path)
        if self.Parameters.script_binary:
            binary_path = str(sdk2.ResourceData(self.Parameters.script_binary).path)
        else:
            binary_path = pj(package, 'tools/yt_run_archive')

        args = [
            binary_path,
            "--yt-proxy", self.Parameters.yt_cluster_name,
            "--package_path", package,
            "--clients", clients_list,
            "--yt-path", yt_path,
            "--ttl-hours", str(self.Parameters.hours),
            "--end-timestamp", str(self.Parameters.last_timestamp),
        ]
        if self.Parameters.write_to_stable:
            args.append("--stable")
        if self.Parameters.write_to_prestable:
            args.append("--prestable")
        if self.Parameters.default_time_segment:
            args.append("--default-time-segment")

        with sdk2.helpers.ProcessLog(self, logger='script_out') as pl:
            sdk2.helpers.subprocess.check_call(
                args,
                env={'YT_TOKEN': yt_token, 'TVM_SECRET': self.Parameters.tvm_token.data()},
                stdout=pl.stdout,
                stderr=pl.stderr
            )

        if self.Parameters.write_archive:
            self.write_archive_table(yt_token)

        return
