import os
import json
import urllib2

from sandbox.projects.RunNewsLoadtest import RunNewsLoadtest
from sandbox.projects.news.RunNewsLoadtest.v2 import RunNewsLoadtestV2
from sandbox.projects.common.news import its_manipulation
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.common.types import task as ctt
from sandbox.common.errors import TaskError

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog

import sandbox.common.errors as ce


LOADTESTING_CONFIGS = {
    "NEWS_WEEKLY": "yweb/news/runtime_scripts/load_testing/robots/complex/weekly.json",
    "NEWS_TESTING": "yweb/news/runtime_scripts/load_testing/robots/complex/testing.json",
}


class RunNewsComplexLoadtests(sdk2.Task):

    environment = (
        environments.SvnEnvironment(),
    )

    class Requirements(sdk2.Task.Requirements):
        disk_space = 15 * 1024
        ram = 10 * 1024

    class Context(sdk2.Context):
        original_its_value = ""
        juggler_mute_id = ""
        testing_config = "void"
        available_tanks = "void"
        stopped_tasks = []
        skip_wait = False
        only_stop = False
        tanks_started = False

    class Parameters(sdk2.Task.Parameters):
        description = "Task for performing full news performance load testing regression"

        generate_only = sdk2.parameters.Bool(
            'Only generate data, don\'t actually shoot'
        )

        mess_with_its = sdk2.parameters.Bool(
            'Disable all neighbour services through ITS (and restore flags after successfull finish)'
        )

        mute_alerts = sdk2.parameters.Bool(
            'Disable known juggler checks'
        )

        lunapark_task = sdk2.parameters.String(
            'ST task ID',
            default_value="NEWS-9407",
            required=False
        )

        target = sdk2.parameters.String(
            'Tank target - host:port',
            default_value="127.1.1.1:80",
            required=False
        )

        with sdk2.parameters.String("Load testing config:") as loadtesting_config:
            loadtesting_config.values.NEWS_WEEKLY = loadtesting_config.Value("Weekly load testing", default=True)
            loadtesting_config.values.NEWS_TESTING = loadtesting_config.Value("Testing load testing (same as weekly, but with low rps", default=False)

        tanks = sdk2.parameters.List(
            'Tanks to shoot with, MTN hostnames',
            required=True
        )

        tankapi_port = sdk2.parameters.String(
            'Tank port, will be the same for all tanks (limitation of tankapi-cmd)',
            default_value="84",
            required=True
        )

        with sdk2.parameters.String('Version load test:') as load_test_version:
            load_test_version.values.v1 = load_test_version.Value('v1', default=True)
            load_test_version.values.v2 = load_test_version.Value('v2')

    def disableNeighbours(self):
        its_manipulation.CheckL7ItsWeights("news.yandex.ru", "news")
        its_manipulation.CheckL7ItsWeights("data.news.yandex.ru", "api")
        self.Context.original_its_value, original_its_version = its_manipulation.GetItsData()
        self.Context.save()
        print("Original version: " + original_its_version)
        print("Original value: " + self.Context.original_its_value)

        # flags won't be available at runtime, if task would be stopped by user
        with open("original_its_value.txt", "w") as text_file:
            text_file.write(self.Context.original_its_value)

        new_its_value = json.loads(self.Context.original_its_value)
        if "_master" not in new_its_value["flags"]:
            new_its_value["flags"]["_master"] = {}

        load_testing_flags = its_manipulation.GetLoadTestingFlags()
        for flag, value in load_testing_flags.items():
            if flag in new_its_value["flags"]["_master"]:
                self.Context.original_its_value = ""  # won't rollback flags after stopping
                self.Context.save()
                raise ce.TaskError("You can't have 2 tasks with mess_with_its flag running simultaniously. If no other "
                                   "RUN_NEWS_LOADTEST/RUN_NEWS_COMPLEX_LOADTEST is running, you must revert its flags manually right now!")
            new_its_value["flags"]["_master"][flag] = value

        print("Modifying to " + json.dumps(new_its_value, indent=4, sort_keys=True))
        self.set_info("Pushing new flags to its. See executor.out log for details")
        its_manipulation.SetItsData(json.dumps(new_its_value, indent=4, sort_keys=True), original_its_version)

    def finishLoadTesting(self):
        # Just in case it will try to run on_execute after waiting for a task
        self.Context.only_stop = True
        self.Context.save()
        for child in self.find():
            if child.id not in self.Context.stopped_tasks:
                self.set_info(
                    "Stopping subtask {}".format(child)
                )
                # If task is already finished it will throw exception and there is no good way to possibility to stop before execution
                try:
                    child.stop()
                except TaskError:
                    pass

                self.set_info(
                    "Wait subtask {} to stop. If it takes too long, maybe you want to stop it manually".format(child)
                )
                self.Context.stopped_tasks.append(child.id)
                self.Context.save()
                if not self.Context.skip_wait:
                    raise sdk2.WaitTask(child.id, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)

        if self.Parameters.mess_with_its:
            try:
                # without version we can't push anything to its
                value, version = its_manipulation.GetItsData()
                print("Read original its value: " + self.Context.original_its_value)
                if self.Context.original_its_value:
                    self.set_info("Reverting its flags to original")
                    its_manipulation.SetItsData(self.Context.original_its_value, version)
            except Exception as e:
                self.set_info("Flags are not reverted!!! Caught exception during its flags push: " + str(e))

        if self.Parameters.mute_alerts and self.Context.juggler_mute_id != "":
            try:
                its_manipulation.FinishJugglerMute(self.Context.juggler_mute_id)
                self.set_info("Juggler mute was finished successfully")
            except:
                self.set_info("Juggler mute was not finished successfully. Please feel free to do it manually")

    def on_execute(self):
        if self.Context.testing_config == "void":
            self.get_loadtesting_config()
            self.Context.available_tanks = self.Parameters.tanks
            self.Context.save()
            if self.Parameters.mess_with_its:
                self.disableNeighbours()

        if self.Parameters.mute_alerts and self.Context.juggler_mute_id == "":
            self.Context.juggler_mute_id = its_manipulation.SetJugglerMute(durationSeconds=3600)
            self.Context.save()
            self.set_info(
                "Juggler mute set at <a href={link}>{link}</a>".format(
                    link="https://juggler.yandex-team.ru/mutes/?query=mute_id={}".format(self.Context.juggler_mute_id)),
                do_escape=False
            )

        while len(self.Context.testing_config) > 0 and not self.Context.only_stop:
            loadtest = self.Context.testing_config[0]
            del self.Context.testing_config[0]
            self.Context.save()

            if len(self.Context.available_tanks) < loadtest["tanks"]:
                self.Context.save()
                raise Exception("Not enouth tanks to run load test")

            tanks_to_use = self.Context.available_tanks[:loadtest["tanks"]]
            self.Context.available_tanks = self.Context.available_tanks[loadtest["tanks"]:]
            self.Context.tanks_started = True
            self.Context.save()

            task_cls = RunNewsLoadtest if self.Parameters.load_test_version == 'v1' else RunNewsLoadtestV2
            sub_task = task_cls(
                self,
                description="Part of complex testing ({})".format(str(loadtest["name"])),
                generate_only=self.Parameters.generate_only,
                mess_with_its=False,
                enable_regression=loadtest["enable_regression"],
                ammo_generator=loadtest["ammo_generator"],
                stop_on_5xx=loadtest["stop_on_5xx"],
                target=self.Parameters.target,
                lunapark_task=self.Parameters.lunapark_task,
                rps_schedule=loadtest["rps_schedule"],
                job_name=loadtest["job_name"],
                do_not_aggregate_res=loadtest["disable_logs_aggregation"],
                tanks=tanks_to_use,
                tankapi_port=self.Parameters.tankapi_port,
                notifications=self.Parameters.notifications,
                create_sub_task=False
            )

            message = """Starting subtask with params:
                      Name: {}
                      Tanks: {}
                      Enable_regression: {}
                      Ammo generator: {}
                      rps_schedule: {}
                      job_name: {}
                      mess_with_its: {}"""
            self.set_info(
                message.format(
                    str(loadtest["name"]),
                    tanks_to_use,
                    loadtest["enable_regression"],
                    loadtest["ammo_generator"],
                    loadtest["rps_schedule"],
                    loadtest["job_name"],
                    loadtest["mess_with_its"],
                )
            )
            sub_task.enqueue()

            self.set_info(
                "Starting subtask {} for {}".format(sub_task, str(loadtest["name"]))
            )

            if loadtest["wait_end"]:
                self.Context.available_tanks = self.Context.available_tanks + tanks_to_use
                self.Context.save()
                raise sdk2.WaitTask(sub_task.id, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)
            else:
                self.Context.save()
                if 'wait_before_next' in loadtest:
                    self.set_info("Sleep for {} seconds".format(loadtest['wait_before_next']))
                    os.system("sleep {}".format(loadtest['wait_before_next']))
        self.finishLoadTesting()

    def on_failure(self, prev_status):
        self.set_info("on_failure called. Looks like something bad happend. Stopping all tanks, reverting flags")
        self.emergency_stop()

    def on_break(self, prev_status, status):
        self.set_info("on_break called (user pressed stop in sandbox?).  Terminating all tanks...")
        self.Context.skip_wait = True
        self.Context.save()
        self.emergency_stop()

    def emergency_stop(self):
        if self.Context.tanks_started:
            for tank in self.Parameters.tanks:
                try:
                    full_url = "http://" + tank + ":" + self.Parameters.tankapi_port + "/api/v1/tests/stop.json"
                    self.set_info("stopping tank " + tank + " reponse: " + urllib2.urlopen(full_url, timeout=5).read())
                except:
                    self.set_info("can't stop tank " + tank + " try running following url manually: " + full_url)

            os.system("sleep 1m")

        self.finishLoadTesting()

    def get_loadtesting_config(self):
        with ProcessLog(self, 'bash_magic.log') as pl:
            with arcadia_sdk.mount_arc_path("arcadia-arc:/#trunk") as path_arc:
                if self.Parameters.loadtesting_config not in LOADTESTING_CONFIGS:
                    raise ce.TaskError('Loadtesting config "{}" is unknown'.format(self.Parameters.loadtesting_config))

                with open(path_arc + "/" + LOADTESTING_CONFIGS[self.Parameters.loadtesting_config]) as conf:
                    self.Context.testing_config = json.load(conf)

                print("Full loadtesting config: " + str(json.dumps(self.Context.loadtesting_config, indent=4)))
