import json
import os
import time
import urllib2
from datetime import date

from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.projects.resource_types import NEWS_AMMO, OTHER_RESOURCE
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog

from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.news import its_manipulation

AMMO_GENERATORS = {
    "NEWS_BACKGROUND": "yweb/news/runtime_scripts/load_testing/robots/collect_background_requests_for_robot-ynews.sh",
    "NEWS_STORIES": "yweb/news/runtime_scripts/load_testing/robots/collect_top_stories_for_robot-ynews.sh",
    "NEWS_STORIES_5XX": "yweb/news/runtime_scripts/load_testing/robots/collect_top_stories_for_robot-ynews_with_5xx.sh",
    "SPORT_MAIN": "yweb/news/runtime_scripts/load_testing/robots/generate_sport_main_ammo_for_robot-ynews.sh",
    "NEWS_STORIES_TOUCH": "yweb/news/runtime_scripts/load_testing/robots/collect_top_stories_touch_for_robot-ynews.sh",
}


class BaseRunNewsLoadtest(sdk2.Task):

    environment = (
        environments.SvnEnvironment(),
    )

    class Context(sdk2.Context):
        original_its_value = ""
        tanks_started = False
        juggler_mute_id = ""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 15 * 1024
        ram = 10 * 1024

    class Parameters(sdk2.Task.Parameters):
        description = "Task for performing News stack load testing with Ya.Tank"

        generate_only = sdk2.parameters.Bool(
            'Only generate data, don\'t actually shoot'
        )

        mess_with_its = sdk2.parameters.Bool(
            'Disable all neighbour services through ITS (and restore flags after successfull finish)'
        )

        mute_alerts = sdk2.parameters.Bool(
            'Disable known juggler checks'
        )

        enable_regression = sdk2.parameters.Bool(
            'Enable regression at lunapark.yandex-team.ru/regress/NEWS',
            default_value=False
        )

        with sdk2.parameters.String("Ammo generator:") as ammo_generator:
            ammo_generator.values.CUSTOM_AMMO = ammo_generator.Value("Custom ammo", default=True)
            ammo_generator.values.NEWS_BACKGROUND = "News background ammo"
            ammo_generator.values.NEWS_STORIES = "News top-10 stories"
            ammo_generator.values.NEWS_STORIES_TURBO = "News top-10 stories. Turbo rendering"
            ammo_generator.values.NEWS_STORIES_5XX = "News top-10 stories. With srcrwr=NEWS_SLAVE_NEWSD:localhost:1234"
            ammo_generator.values.NEWS_STORIES_TOUCH = "News top-10 stories. Touch"
            ammo_generator.values.SPORT_MAIN = "Sport main page"
            ammo_generator.values.MIRROR_MAIN = "Mirror main page"

        with ammo_generator.value["CUSTOM_AMMO"]:
            precooked_ammo = sdk2.parameters.Resource(
                'Custom ammo to shoot with',
                required=False
            )

        stop_on_5xx = sdk2.parameters.Bool(
            'Stop, if there are more than 5% of 5xx in last 10 seconds',
            default_value=False
        )

        do_not_aggregate_res = sdk2.parameters.Bool(
            'Do not aggregate results after load testing',
            default_value=False
        )

        target = sdk2.parameters.String(
            'Nanny service or several hosts comma-separated',
            default_value="rtc_balancer_news_production_knoss_sas",
            required=True
        )

        target_port = sdk2.parameters.String(
            'Target port',
            default_value="80",
            required=True
        )

        lunapark_task = sdk2.parameters.String(
            'ST task ID',
            default_value="NEWS-9407",
            required=False
        )

        rps_schedule = sdk2.parameters.String(
            'RPS schedule',
            default_value='line(1,1000,5m)',
            required=False
        )

        job_name = sdk2.parameters.String(
            'Job name without spaces. Use something like story_linear_mix_7_3_0319_nof_0.3. Today\'s date will be added automatically',
            default_value='standard_news_load_test',
            required=False
        )

        with sdk2.parameters.String("Tank logging:") as tank_logging:
            tank_logging.values.NORMAL = tank_logging.Value("Usual logging. Recommended", default=True)
            tank_logging.values.ERRORS_5xx = "Log all 5xx. Use if there are some 5xx requests you wish to debug. Decreasing (2x, may be 5x) tank capacity"
            tank_logging.values.ERRORS_5xx_4xx = "Log all 5xx & 4xx. Use with all cautions. Decreasing (2x, may be 5x) tank capacity"
            tank_logging.values.EVERYTHING = "Log EVERYTHING. Only when you know what are you doing. Significantly (10x, may be 100x) decreasing tank capacity"

        tanks = sdk2.parameters.List(
            'Tanks to shoot with, MTN hostnames',
            required=True
        )

        tankapi_port = sdk2.parameters.String(
            'Tank port, will be the same for all tanks (limitation of tankapi-cmd)',
            default_value="80",
            required=True
        )

        with lunapark_task.value["NEWS-14467"]:
            additional_tank_config = sdk2.parameters.String(
                'These lines will be added to resulting tank config',
                required=False,
                default='',
                multiline=True
            )

    def on_execute(self):
        self._check_tanks()
        config_resource = self._get_load_config(self._get_ammo_url())
        self._mute_alerts_if_need()
        self._its_weights_modify_if_need()
        self._run_load_test(config_resource)
        self._finish_load_test()

    def on_failure(self, prev_status):
        self.set_info("on_failure called. Looks like something bad happend. Stopping all tanks, reverting flags")
        self.emergency_stop()

    def on_break(self, prev_status, status):
        self.set_info("on_break called (user pressed stop in sandbox?).  Terminating all tanks...")
        self.emergency_stop()

    def emergency_stop(self):
        if self.Context.tanks_started:
            for tank in self.Parameters.tanks:
                try:
                    full_url = "http://" + tank + ":" + self.Parameters.tankapi_port + "/api/v1/tests/stop.json"
                    self.set_info("stopping tank " + tank + " reponse: " + urllib2.urlopen(full_url, timeout=5).read())
                except:
                    self.set_info("can't stop tank " + tank + " try running following url manually: " + full_url)

        self._finish_load_test()

    def _run_load_test(self, config_resource):
        if not self.Parameters.generate_only:
            tank_opts = []
            if len(self.Parameters.tanks) > 1:
                tank_opts.append('-m')

            for tank in self.Parameters.tanks:
                tank_opts.extend(['-t', tank])

            tank_opts.extend(['-p', self.Parameters.tankapi_port])

            if len(self.Parameters.additional_tank_config) > 0:
                tank_opts.extend(['--patch-cfg', self.Parameters.additional_tank_config])

            try:
                self.Context.tanks_started = True
                self.Context.save()
                first_try = time.time()
                while time.time() < first_try + 15:  # withing 15 seconds making retries, if tankapi returns too quickly
                    if time.time() > first_try + 1:
                        self.set_info("Some problems with tankapi-cmd, making another try")
                    p = sp.Popen(
                        ['tankapi-cmd', '-c', str(config_resource.path)] + tank_opts,
                        stdout=sp.PIPE
                    )
                    while p.poll() is None:
                        time.sleep(0.1)
                        line = p.stdout.readline().decode('utf-8')
                        if line != '':
                            print(line)
                        if 'Web link: https://lunapark.yandex-team.ru' in line:
                            link = line.split('\n')[0].split(' ')[-1]
                            self.set_info(
                                "Look for realtime load at <a href={link}>{link}</a>".format(link=link),
                                do_escape=False
                            )
                            print(p.stdout.read().decode('utf-8'))
                    time.sleep(2)
            except sp.CalledProcessError as e:
                print('tankapi-cmd finished with error: {e}'.format(e=str(e)))

    def _get_load_config(self, ammo_url):
        raise NotImplementedError()

    def _check_tanks(self):
        for tank in self.Parameters.tanks:
            try:
                full_url = "http://" + tank + ":" + self.Parameters.tankapi_port + "/api/v1/tank/status.json"
                tank_response = urllib2.urlopen(full_url, timeout=5).read()
                if json.loads(tank_response)["is_testing"] is True:
                    raise ce.TaskError("Looks like tank " + tank + " is running right now. Is another RUN_NEWS_LOADTEST in process? Stopping...")
            except ce.TaskError:
                raise
            except:
                raise ce.TaskError("Looks like tank " + tank + " is not responding. Please replace it with working one. Stopping... ")

    def _mute_alerts_if_need(self):
        if self.Parameters.mute_alerts:
            self.Context.juggler_mute_id = its_manipulation.SetJugglerMute(durationSeconds=3600)
            self.Context.save()
            self.set_info(
                "Juggler mute set at <a href={link}>{link}</a>".format(
                    link="https://juggler.yandex-team.ru/mutes/?query=mute_id={}".format(self.Context.juggler_mute_id)),
                do_escape=False
            )

    def _get_ammo_url(self):
        ammo_res = self._get_ammo_resource()
        return "https://proxy.sandbox.yandex-team.ru/" + str(ammo_res.id)

    def _get_ammo_resource(self):
        if not self.Parameters.precooked_ammo:
            with ProcessLog(self, 'ammo_generator') as pl:
                try:
                    with sdk2.ssh.Key(self, "NEWS", "ssh_key"):
                        assert os.environ["SSH_AUTH_SOCK"]
                        assert os.environ["SSH_AGENT_PID"]
                        with arcadia_sdk.mount_arc_path("arcadia-arc:/#trunk") as path_arc:
                            cwd_backup = os.path.abspath(os.getcwd())
                            os.chdir(path_arc)

                            if self.Parameters.ammo_generator not in AMMO_GENERATORS:
                                raise ce.TaskError('Ammo generator "{}" is unknown'.format(self.Parameters.ammo_generator))

                            sp.check_call(
                                ["bash", "-c", AMMO_GENERATORS[self.Parameters.ammo_generator] + " > {}/res.ammo || true".format(cwd_backup)],
                                stdout=pl.stdout, stderr=pl.stderr
                            )

                            # next shit removes last incomplete ammo. This needs to be in bash, because I will never debug out python in sandbox
                            for i in range(1, 100):
                                sp.check_call(
                                    ["bash", "-c",
                                     "[ -z \"$(tail -n 1 {cwd}/res.ammo | cut -c 4-)\" ] || sed -i \"\\$d\" {cwd}/res.ammo".format(cwd=cwd_backup)],
                                    stdout=pl.stdout, stderr=pl.stderr
                                )

                            os.chdir(cwd_backup)
                            ammo_res = NEWS_AMMO(self, "Ammofile", "res.ammo")
                            sdk2.ResourceData(ammo_res).ready()
                            return ammo_res

                except sp.CalledProcessError as e:
                    raise ce.TaskError('Error generating ammo: {e}'.format(e=str(e)))
        else:
            return self.Parameters.precooked_ammo

    def _its_weights_modify_if_need(self):
        if self.Parameters.mess_with_its:
            its_manipulation.CheckL7ItsWeights("news.yandex.ru", "news")
            its_manipulation.CheckL7ItsWeights("data.news.yandex.ru", "api")
            self.Context.original_its_value, original_its_version = its_manipulation.GetItsData()
            self.Context.save()
            print("Original version: " + original_its_version)
            print("Original value: " + self.Context.original_its_value)

            # flags won't be available at runtime, if task would be stopped by user
            with open("original_its_value.txt", "w") as text_file:
                text_file.write(self.Context.original_its_value)

            new_its_value = json.loads(self.Context.original_its_value)
            if "_master" not in new_its_value["flags"]:
                new_its_value["flags"]["_master"] = {}

            load_testing_flags = its_manipulation.GetLoadTestingFlags()
            for flag, value in load_testing_flags.items():
                if flag in new_its_value["flags"]["_master"]:
                    self.Context.original_its_value = ""  # won't rollback flags after stopping
                    self.Context.save()
                    raise ce.TaskError("You can't have 2 tasks with mess_with_its flag running simultaniously. If no other "
                                       "RUN_NEWS_LOADTEST is running, you must revert its flags manually right now!")
                new_its_value["flags"]["_master"][flag] = value

            print("Modifying to " + json.dumps(new_its_value, indent=4, sort_keys=True))
            self.set_info("Pushing new flags to its. See executor.out log for details")
            its_manipulation.SetItsData(json.dumps(new_its_value, indent=4, sort_keys=True), original_its_version)
            os.system("sleep 1m")

    def _finish_load_test(self):
        if self.Parameters.mess_with_its:
            try:
                # without version we can't push anything to its
                value, version = its_manipulation.GetItsData()
                print("Read original its value: " + self.Context.original_its_value)
                if self.Context.original_its_value:
                    self.set_info("Reverting its flags to original")
                    its_manipulation.SetItsData(self.Context.original_its_value, version)
            except Exception, e:
                self.set_info("Flags are not reverted!!! Caught exception during its flags push: " + str(e))

        if self.Parameters.mute_alerts and self.Context.juggler_mute_id != "":
            try:
                its_manipulation.FinishJugglerMute(self.Context.juggler_mute_id)
                self.set_info("Juggler mute was finished successfully")
            except:
                self.set_info("Juggler mute was not finished successfully. Please feel free to do it manually")

        if not self.Context.tanks_started:
            self.set_info("finishing load testing and not started tanks? Discarding all data")
            return

        if self.Parameters.do_not_aggregate_res:
            self.set_info("finishing load testing without aggregating results")
            return

        tank_phout_res = []
        with sdk2.ssh.Key(self, "NEWS", "ssh_key"):
            assert os.environ["SSH_AUTH_SOCK"]
            assert os.environ["SSH_AGENT_PID"]
            finish_timestamp = int(time.time())  # last sent request by first stopped tank
            for tank in self.Parameters.tanks:
                try:
                    print('Using tank {} for united results'.format(tank))
                    current_phout = sp.check_output(
                        ['ssh',
                         '-o',
                         'StrictHostKeyChecking=no',
                         'robot-ynews@' + tank,
                         'cat 2>/dev/null `find /* 2>/dev/null -printf "%T+\t%p\n" | grep logs | grep yandex-tank | grep phout | sort | tail -n 1` || true'
                         ]).split('\n')

                    if len(current_phout) > 10:
                        max_local_timestamp = 0
                        for record in current_phout:
                            try:
                                max_local_timestamp = max(max_local_timestamp, int(record.split('.')[0]) + 1)
                            except:
                                pass

                        finish_timestamp = min(finish_timestamp, max_local_timestamp)

                        tank_phout_res += current_phout
                    else:
                        self.set_info("Load testing log for tank " + tank + " is not available. See full logs in resources for details")

                    tank_logs = sp.check_output(
                        ['ssh',
                         '-o',
                         'StrictHostKeyChecking=no',
                         'robot-ynews@' + tank,
                         'tar -czf - `find /* -type d 2>/dev/null -printf "%T+\t%p\n" | grep logs | grep yandex-tank | sort | tail -n 1` 2>/dev/null || true'
                         ])

                    logs_name = tank + ".tar.gz"
                    with open(logs_name, 'w') as fd:
                        fd.write(tank_logs)

                    tank_logs_resource = OTHER_RESOURCE(self, "Full logs for tank " + tank, logs_name)
                    sdk2.ResourceData(tank_logs_resource).ready()
                except sp.CalledProcessError as e:
                    self.set_info("ignoring tank {} - {}. Aggregation might be invalid, but sandbox task should not be restarted here!".format(tank, str(e)))

            for i in range(-10, 1):
                tank_phout_res += [str(finish_timestamp + i) + ".833\tgood\t120401\t78\t18\t117324\t2981\t120091\t384\t49344\t0\t205"]  # HTTP 205 - testing is over

            tank_phout_cutted = []
            for record in tank_phout_res:
                try:
                    if int(record.split('.')[0]) < finish_timestamp:
                        tank_phout_cutted += [record]
                except:
                    pass

            tank_phout_cutted.sort()
            print(len(tank_phout_cutted))
            print(len('\n'.join(tank_phout_cutted)))

            if self.Parameters.enable_regression:
                self.set_info("Look for all similiar tests at https://lunapark.yandex-team.ru/regress/NEWS")

            # we don't need to upload united results, if the tank was alone
            if len(self.Parameters.tanks) > 1:
                subprocess = sp.Popen(['ssh', '-o', 'StrictHostKeyChecking=no', 'robot-ynews@' + self.Parameters.tanks[0], 'cat > /tmp/phout.log'], stdout=sp.PIPE, stdin=sp.PIPE)
                subprocess.communicate('\n'.join(tank_phout_cutted))
                subprocess.stdin.close()

                with open('magic.yaml', 'w') as fd:
                    fd.write('phantom:\n')
                    fd.write('  enabled: false\n')
                    fd.write('pandora:\n')
                    fd.write('  enabled: false\n')
                    fd.write('ShootExec:\n')
                    fd.write('  cmd: cp /tmp/phout.log /tmp/phout.log.copy\n')
                    fd.write('  package: yandextank.plugins.ShootExec\n')
                    fd.write('  output_path: /tmp/phout.log.copy\n')
                    fd.write('  enabled: true\n')
                    fd.write('uploader:\n')
                    fd.write('  enabled: true\n')
                    if self.Parameters.enable_regression:
                        fd.write('  component: ' + self.Parameters.job_name + '\n')
                    fd.write('  job_name: ' + str(date.today()) + '_' + self.Parameters.job_name + '\n')
                    fd.write('  lock_targets: []\n')
                    fd.write('  operator: robot-ynews\n')
                    fd.write('  package: yandextank.plugins.DataUploader\n')
                    fd.write('  task: ' + self.Parameters.lunapark_task + '\n')
                    fd.write('autostop:\n')
                    fd.write("  autostop: ['http(205, 1, 10s)']\n")
                    fd.write('  enabled: true\n')
                    fd.write('  package: yandextank.plugins.Autostop\n')

                magic_yaml_resource = OTHER_RESOURCE(self, "Generated magic.yaml", "magic.yaml")
                sdk2.ResourceData(magic_yaml_resource).ready()

                sp.check_output(['scp', '-o', 'StrictHostKeyChecking=no', 'magic.yaml', 'robot-ynews@' + self.Parameters.tanks[0] + ':/tmp/'])

                output = sp.check_output(['ssh', '-o', 'StrictHostKeyChecking=no', 'robot-ynews@' + self.Parameters.tanks[0], 'yandex-tank', '-c', '/tmp/magic.yaml'])
                print(output)
                for line in output.split('\n'):
                    if 'link' in line:
                        self.set_info(line)
                        break
