import re
import yaml
import requests
import json
from datetime import date

from sandbox import sdk2
from sandbox.common import errors as ce
from sandbox.projects.resource_types import OTHER_RESOURCE

from sandbox.projects.news.RunNewsLoadtest.base import BaseRunNewsLoadtest

DEFAULT_RPS_SCHEDULE = [
    {
        'type': 'line',
        'duration': '10m',
        'from': 1,
        'to': 6000,
    }
]

RE_SCHEDULE = re.compile(r'(?P<type>(line|const))\(\s*(?P<schedule>[\d\s,ms]+)\s*\)')
RE_PROFILE = re.compile(r'\d+[ms]?')

AMMO_FILE_DST = "./ammo.txt"


class RunNewsLoadtestV2(BaseRunNewsLoadtest):
    class Parameters(BaseRunNewsLoadtest.Parameters):
        container = sdk2.parameters.Container(
            'LXC Container',
            default_value=2240022105,  # tankapi-cmd
            required=True
        )

    def _resolve_service(self, service):
        full_response = requests.get("http://nanny.yandex-team.ru/v2/services/{}/runtime_attrs/instances/".format(service))
        response = json.loads(full_response.text)
        hosts = []
        for pod in response["content"]["yp_pod_ids"]["pods"]:
            if pod["cluster"] == "SAS":
                hosts.append(pod["pod_id"] + ".sas.yp-c.yandex.net")
        if not hosts:
            raise "Hosts not resolved for endpointset {}".format(service)
        return hosts

    def _get_load_config(self, ammo_url):
        load_config = {
            "shellexec": {
                "enabled": True,
                "end": "rm -rf /tmpfs/*.stpd",
                "package": "yandextank.plugins.ShellExec"
            },
            "phantom": {"enabled": False},
            "pandora": {
                "enabled": True,
                "expvar": True,
                "package": "yandextank.plugins.Pandora",
                "pandora_cmd": "/usr/local/bin/pandora",
                "resources": [
                    {
                        "src": ammo_url,
                        "dst": AMMO_FILE_DST,
                    }
                ],
                "config_content": {
                    "log": {"level": "error"},
                    "monitoring": {
                        "expvar": {"enabled": True, "port": 1234},
                    },
                    "pools": [
                    ],
                }
            }
        }

        target = self.Parameters.target
        if ',' in target or '.' in target:
            hosts = target.split(',')
        else:
            hosts = self._resolve_service(self.Parameters.target)

        for host in hosts:
            load_config["pandora"]["config_content"]["pools"].append({
                "ammo": {"file": AMMO_FILE_DST, "type": "raw"},
                "gun": {"target": host + ":" + self.Parameters.target_port, "type": "http"},
                "id": "HTTP",
                "result": {"destination": "./phout.{}.log".format(host), "type": "phout"},
                "rps": self._parsed_rps_schedule(len(hosts)),
                "startup": [{"times": 20, "type": "once"}],
            })

        if self.Parameters.stop_on_5xx:
            load_config["autostop"] = {
                "autostop": ['http(5xx, 5%, 10s)', 'net(1xx, 5%, 10s)', 'http(204, 5%, 10s)', 'time(1500, 10s)'],
                "enabled": True,
                "package": "yandextank.plugins.Autostop",
            }

        load_config["uploader"] = {
            "api_address": "https://lunapark.yandex-team.ru/",
            "enabled": True,
            "operator": "robot-ynews",
            "package": "yandextank.plugins.DataUploader",
        }

        if len(self.Parameters.tanks) > 1:
            self.set_info('Look for realtime load at https://lunapark.yandex-team.ru/' + self.Parameters.lunapark_task)
            load_config["uploader"]["job_name"] = "news_regular_loadtest"
        else:
            load_config["uploader"]["job_name"] = str(date.today()) + '_' + self.Parameters.job_name
            if self.Parameters.enable_regression:
                load_config["uploader"]["component"] = self.Parameters.job_name

        if self.Parameters.lunapark_task:
            load_config["uploader"]["task"] = self.Parameters.lunapark_task

        with open('load.yaml', 'w') as fd:
            yaml.safe_dump(load_config, fd, default_flow_style=False, encoding='utf-8', allow_unicode=True)

        load_config_resource = OTHER_RESOURCE(self, "Generated load.yaml", "load.yaml")
        sdk2.ResourceData(load_config_resource).ready()

        return load_config_resource

    def _parsed_rps_schedule(self, instance_count):
        if not self.Parameters.rps_schedule:
            return DEFAULT_RPS_SCHEDULE

        rps_schedule = str(self.Parameters.rps_schedule)
        result = []
        for schedule in RE_SCHEDULE.finditer(rps_schedule):
            schedule_dict = schedule.groupdict()
            type_ = schedule_dict['type']
            args = RE_PROFILE.findall(schedule_dict['schedule'])

            if type_ == 'line' and len(args) == 3:
                from_, to_, duration = args
                result.append(
                    {
                        'type': type_,
                        'from': int(from_) / instance_count,
                        'to': int(to_) / instance_count,
                        'duration': duration,
                    })
            elif type_ == 'const' and len(args) == 2:
                ops, duration = args
                result.append(
                    {
                        'type': type_,
                        'ops': int(ops) / instance_count,
                        'duration': duration,
                    })
            else:
                raise ce.TaskError('Error config parsing: {}'.format(rps_schedule))

        return result
