import os
import json
import copy

from sandbox.projects.RunNewsLoadtest import RunNewsLoadtest
from sandbox.common.types import task as ctt

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog

from sandbox.projects.resource_types import OTHER_RESOURCE


class RunNewsRegressLoadtests(sdk2.Task):

    environment = (
        environments.SvnEnvironment(),
    )

    class Requirements(sdk2.Task.Requirements):
        disk_space = 15 * 1024
        ram = 10 * 1024

    class Context(sdk2.Context):
        regress_config = "void"

    class Parameters(sdk2.Task.Parameters):
        description = "Task for performing full news performance load testing regression"

        tank = sdk2.parameters.String(
            'Tank to shoot with, MTN hostname',
            default_value="sas1-5594-sas-news-load-balancer-15090.gencfg-c.yandex.net",
            required=True
        )

        tankapi_port = sdk2.parameters.String(
            'Tank port',
            default_value="15094",
            required=True
        )

    def on_execute(self):
        if self.Context.regress_config == "void":
            self.generate_ammo()
            self.generate_const_load()
            self.Context.save()
            print("Full regress config: " + str(json.dumps(self.Context.regress_config, indent=4)))

        while len(self.Context.regress_config) > 0:
            service = self.Context.regress_config[0]
            del self.Context.regress_config[0]
            self.Context.save()

            yasm_config_part = ""
            if "yasm_tags" in service:
                yasm_config_part += "yasm:\n"
                yasm_config_part += "  enabled: true\n"
                yasm_config_part += "  package: yandextank.plugins.YASM\n"
                yasm_config_part += "  panels:\n"
                yasm_config_part += "    panel1:\n"
                yasm_config_part += "      host: " + service["host"].split(":")[0] + "\n"
                yasm_config_part += "      tags: " + service["yasm_tags"] + "\n"

            test_type = "load test"
            if "const_load" in service["lunapark_name"]:
                test_type = "const load test"

            sub_task = RunNewsLoadtest(
                self,
                description="regression {} for service {}".format(test_type, str(service["service"])),
                generate_only=False,
                mess_with_its=False,
                enable_regression=True,
                precooked_ammo=service["ammo"],
                stop_on_5xx=True,
                target=service["host"],
                lunapark_task="NEWS-14467",
                rps_schedule=service["custom_schedule"],
                job_name=service["lunapark_name"],
                tanks=[self.Parameters.tank],
                tankapi_port=self.Parameters.tankapi_port,
                additional_tank_config=yasm_config_part,
                notifications=self.Parameters.notifications,
                create_sub_task=False
            )

            sub_task.enqueue()

            self.set_info(
                "Starting subtask {} for service {}".format(sub_task, str(service["service"]))
            )

            raise sdk2.WaitTask(sub_task.id, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)

    def generate_ammo(self):
        with ProcessLog(self, 'bash_magic.log') as pl:
            with sdk2.ssh.Key(self, "NEWS", "ssh_key"):
                assert os.environ["SSH_AUTH_SOCK"]
                assert os.environ["SSH_AGENT_PID"]

                environments.SvnEnvironment().prepare()

                sp.check_call(
                    ["bash", "-c", "svn cat svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya | python - clone"],
                    stdout=pl.stdout, stderr=pl.stderr
                )

                sp.check_call(
                    ["arcadia/ya", "make", "--checkout", "arcadia/yweb/news/runtime_scripts/load_testing"],
                    stdout=pl.stdout, stderr=pl.stderr
                )

                sp.check_call(
                    ["arcadia/ya", "make", "--checkout", "arcadia/apphost/tools/make_tank_ammo"],
                    stdout=pl.stdout, stderr=pl.stderr
                )

                with open('arcadia/yweb/news/runtime_scripts/load_testing/robots/regress/regress_config.json') as conf:
                    self.Context.regress_config = json.load(conf)

                for i in range(len(self.Context.regress_config)):
                    generation_command = "arcadia/yweb/news/runtime_scripts/load_testing/robots/regress/generate_ammo_for_service"
                    if "custom_ammo_generation_command" in self.Context.regress_config[i]:
                        generation_command = self.Context.regress_config[i]["custom_ammo_generation_command"]

                    sp.check_call([
                        generation_command,
                        self.Context.regress_config[i]["service"],
                        self.Context.regress_config[i]["host"]
                    ], stdout=pl.stdout, stderr=pl.stderr)

                    ammo = OTHER_RESOURCE(self, "Ammo for " + str(self.Context.regress_config[i]["service"]), self.Context.regress_config[i]["service"] + ".ammo")
                    sdk2.ResourceData(ammo).ready()
                    self.Context.regress_config[i]["ammo"] = ammo.id
                    if "custom_schedule" not in self.Context.regress_config[i]:
                        self.Context.regress_config[i]["custom_schedule"] = "line(1,2000,7m)"

    def generate_const_load(self):
        for i in range(len(self.Context.regress_config)):
            conf = copy.deepcopy(self.Context.regress_config[i])
            conf["custom_schedule"] = "line(1, 50, 30s) const(50, 2m)"
            conf["lunapark_name"] = conf["lunapark_name"].replace("_performance_test", "") + "_const_load"
            self.Context.regress_config.append(conf)
