# -*- coding: utf-8 -*-
import tempfile

from sandbox import sdk2
from sandbox.sdk2.resource import Resource
from sandbox.sdk2.helpers import subprocess as sp


class BackofficeMongoDump(Resource):
    executable = False


class UpdateNewsBackofficeTestData(sdk2.Task):
    """
    Таск для дампа Mongo админки новостей для тестов
    """

    class Requirements(sdk2.Requirements):
        container_resource = 2406803418

    class Parameters(sdk2.Task.Parameters):

        hosts = sdk2.parameters.List(
            'List of Mongo hosts',
            default=[
                'man-9qbvzix4hic3s8h6.db.yandex.net',
                'sas-zyaa8u0yvwwbygfl.db.yandex.net',
                'vla-9p33ipp2ey613oav.db.yandex.net',
            ],
            required=True,
        )
        database = sdk2.parameters.String('Database name to dump', default='news_back', required=True)
        replica_set = sdk2.parameters.String('Replica set', default='rs01', required=True)
        port = sdk2.parameters.Integer('Database port', default=27018, required=True)
        user = sdk2.parameters.String('Database username', default='news_user', required=True)

        # https://docs.yandex-team.ru/sandbox/dev/secret#howto
        tokens = sdk2.parameters.YavSecret(
            'YAV secret identifier (with optional version)',
            default='sec-01ejwqs1p3gbx97jcvs2r58mqv@ver-01ejwqs1pcce92vndfqym9cd3b'
        )

    def on_execute(self):
        host = (f'{self.Parameters.replica_set}/'
                f'{",".join(f"{host}:{self.Parameters.port}" for host in self.Parameters.hosts)}')
        base_cmd = [
            'mongodump',
            '--host', host,
            '-u', self.Parameters.user,
            '--db', self.Parameters.database,
        ]
        ratings_cmd = base_cmd + [
            '-c', 'rating',
            '-q', '{"is_active": true}',
        ]
        collections_to_exclude = frozenset({'rating', 'rkn_agency', 'rkn_history', 'samsara_ticket', 'api_history'})
        main_cmd = base_cmd + [
            f'--excludeCollection={collection}'
            for collection in collections_to_exclude
        ] + ['--forceTableScan']

        output = 'dump.tar.bz2'
        password = self.Parameters.tokens.data()['password']
        with sdk2.helpers.ProcessLog(self, logger='mongo_dump') as pl:
            run_mongodump(main_cmd, password, pl)
            run_mongodump(ratings_cmd, password, pl)
            sp.check_call(['tar', 'cfj', output, 'dump'], stderr=pl.stderr, stdout=pl.stdout)
            BackofficeMongoDump(self, "News's backoffice Mongo db dump", output)


def run_mongodump(cmd, password: str, process_log):
    with tempfile.TemporaryFile() as fp:
        fp.write(password.encode())
        fp.write(b'\n')
        fp.seek(0)
        sp.check_call(cmd, stderr=process_log.stderr, stdout=process_log.stdout, stdin=fp)
