#!/usr/bin/env python
# -*- coding: utf-8 -*-

import time
import datetime
import cgi
import collections
import jinja2
import logging

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.channel import channel


def to_string(s):
    if isinstance(s, unicode):
        return s.encode('utf-8')
    return str(s)


def to_unicode(s):
    if isinstance(s, str):
        return s.decode('utf-8')
    return unicode(s)


NewsItem = collections.namedtuple("NewsItem", "url title text")


def gen_html(items):
    template = jinja2.Template(u"""
<html>
<meta charset="UTF-8">
<p>Найдено <b>{{ count }}</b> сообщений по запросу</p>
<ol>
{% for item in items %}
  <li>
    <p>
       <a href="{{ item.url }}">{{ item.title }}</a>
    </p>
    <p>{{ item.text }}</p>
  </li>
{% endfor %}
</ol>
<html>
""")
    return template.render(items=items, count=len(items))


OBJECTS = [
    [u'движение', u'о'],
    [u'путепровод', u''],
    [u'развязка', u'а'],
    [u'эстакада', u'а'],
    [u'мост', u''],
    [u'тоннель', u''],
    [u'переправа', u'а'],
    [u'дорога', u'а'],
    [u'магистраль', u'а'],
    [u'проезд', u''],
    [u'акведук', u''],
    [u'переезд', u''],
    [u'зимник', u''],
    [u'брод', u''],
    [u'дублёр', u''],
    [u'перекрёсток', u''],
]

ACTIONS = [
    u'закрыт',
    u'ограничен',
    u'перекрыт',
    u'прекращен',
    u'изменен',
    u'открыт',
    u'пущен',
    u'запущен',
    u'восстановлен',
]


def make_query(objects, actions):
    def obj_query(obj):
        actions_part = u" | ".join(map(lambda a: a + obj[1], actions))
        return u'({0} & ({1}))'.format(obj[0], actions_part)

    return u' | '.join([obj_query(o) for o in objects])


QUERY = make_query(OBJECTS, ACTIONS)

# FIXME pkostikov: get rid of tmp solution with .so file from sandbox
# comes from here https://st.yandex-team.ru/NMAPS-7617

YQL_TEMPLATE = u'''
PRAGMA yt.DefaultMemoryLimit = "4096m";

PRAGMA yt.PoolTrees = "physical";
PRAGMA yt.TentativePoolTrees = "cloud";

PRAGMA file("as", "https://proxy.sandbox.yandex-team.ru/520792074");
PRAGMA udf("as");

$context = AugurSearch2::CreateParserContext();

$query = "{query}";

$titleHits = ($title)  -> {{return AugurSearch2::Search(
        AugurSearch2::CreateSearchIndex(
            AugurSearch2::CreateParsedText($context, $title, "ru" as language)
        ),
        $query
    );
}};

$bodyHits = ($content)  -> {{return AugurSearch2::Search(
        AugurSearch2::CreateSearchIndex(
            AugurSearch2::CreateParsedText($context, $content, "ru" as language)
        ),
        $query
    );
}};

$unique_articles = (SELECT
    url,
    MIN_BY(doc, fetch_time) as doc,
    MIN(fetch_time) as fetch_time
FROM `//home/news/dynstorage/info-russian/data`
WHERE
    fetch_time >= {history_since}
    -- fetch_time >= {history_since_human_readable}
GROUP BY url);


$fresh_articles = (
    SELECT
        url,
        fetch_time,
        doc.PatchedText.Title ?? "" as title,
        doc.PatchedText.Text ?? "" as content
    FROM $unique_articles
    WHERE
        fetch_time >= {since} AND fetch_time < {till}
        -- fetch_time >= {since_human_readable} fetch_time < {till_human_readable}
);


$result = (
    SELECT
        url,
        title,
        content
    FROM $fresh_articles
    WHERE ListLength($titleHits(title)) > 0
          OR ListLength($bodyHits(content)) > 0
);

SELECT * FROM $result;

'''


def hack_row_data(u):
    # tmp workaround due to encoding issues in yql.api.v1.client
    # https://st.yandex-team.ru/ARMDEV-253
    if isinstance(u, unicode):
        if all(ord(c) < 256 for c in u):
            return ''.join([chr(ord(c)) for c in u])
        else:
            return to_string(u)
    assert(isinstance(u, str))
    return u


class NotifyRoadEventsNews(sdk2.Task):

    class Parameters(sdk2.ServiceTask.Parameters):
        yql_vault_token = sdk2.parameters.String("yql_vault_token", required=True)
        emails = sdk2.parameters.List("emails", required=True)
        history_since_days_ago = sdk2.parameters.Integer("days from now to start search duplicates", default=5, required=True)
        since_minutes_ago = sdk2.parameters.Integer("minutes from now to start search", default=40, required=True)
        till_minutes_ago = sdk2.parameters.Integer("minutes from now to end search", default=20, required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
        )

    def on_execute(self):
        now = int(time.time())
        since = now - self.Parameters.since_minutes_ago * 60
        till = now - self.Parameters.till_minutes_ago * 60
        history_since = now - self.Parameters.history_since_days_ago * 60 * 60 * 24

        date_tmpl = '%Y-%m-%d %H:%M:%S UTC'
        till_human_readable = datetime.datetime.utcfromtimestamp(
            till).strftime(date_tmpl)
        since_human_readable = datetime.datetime.utcfromtimestamp(
            since).strftime(date_tmpl)
        history_since_human_readable = datetime.datetime.utcfromtimestamp(
            history_since).strftime(date_tmpl)

        from yql.api.v1.client import YqlClient

        client = YqlClient(
            db='hahn',
            token=sdk2.Vault.data(self.Parameters.yql_vault_token)
        )

        request = client.query(
            YQL_TEMPLATE.format(
                since=since,
                till=till,
                history_since=history_since,
                since_human_readable=since_human_readable,
                till_human_readable=till_human_readable,
                history_since_human_readable=history_since_human_readable,
                query=QUERY),
            syntax_version=1
        )
        request.run()

        items = []
        for table in request.get_results():
            table.fetch_full_data()

            column_name_to_index = {}
            cur_index = 0
            for column_name, _ in table.columns:
                column_name_to_index[column_name] = cur_index
                cur_index += 1

            for row in table.rows:
                items.append(NewsItem(
                    url=to_unicode(hack_row_data(row[column_name_to_index['url']])),
                    title=to_unicode(hack_row_data(row[column_name_to_index['title']])),
                    text=cgi.escape(to_unicode(hack_row_data(row[column_name_to_index['content']]))[:100])
                ))

        logging.info('Fetched {0} items'.format(len(items)))

        if not items:
            return

        html = gen_html(items)
        body = to_string(html)

        logging.info('Sending email with content:\n{0}'.format(body))

        channel.sandbox.send_email(
            mail_to=self.Parameters.emails,
            mail_cc=[],
            mail_subject="Результат запроса в Яндекс.Новости",
            mail_body=body,
            content_type='text/html',
            charset='utf-8'
        )
