#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
from sandbox import sdk2
from sandbox.projects.nmaps.ugc_assignments.PrepareUgcPushes import (
    ADDRESS_ADD,
    SETTLEMENT_SCHEME,
)
from sandbox.projects.nmaps.ugc_assignments.common.resources import (
    MapsAbstractBinaryForPushes,
)
from sandbox.projects.nmaps.ugc_assignments.common.task_utils import (
    check_subtasks,
    get_vault_data,
    ensure_latest_resources,
    wait_tasks,
)
from sandbox.projects.nmaps.ugc_assignments.common.ugc_pushes_parameters import (
    BuildAssignmentsParameters,
)
from sandbox.projects.nmaps.ugc_assignments.ExecTask import (
    MapsExecTask,
)
from sandbox.projects.nmaps.ugc_assignments.ExecRunPythonUdf import (
    MapsExecRunPythonUdf,
)
from sandbox.sandboxsdk.environments import PipEnvironment


UUIDS_PATH_TEMPLATE = "//home/maps-nmaps/analytics/uuids_for_pushes/{{{start_date}..{end_date}}}"


class MapsPrepareBuildingsForAddressPushesBinary(MapsAbstractBinaryForPushes):
    pass


class MapsPrepareDwellplacesForAddressPushesBinary(MapsAbstractBinaryForPushes):
    pass


class MapsMakeAddressPushesBinary(MapsAbstractBinaryForPushes):
    pass


class MakeAddressAndSettlementsAssignmentsPushes(sdk2.Task):
    class Parameters(BuildAssignmentsParameters):
        with sdk2.parameters.RadioGroup("Assignment type") as assignment_type:
            assignment_type.values[ADDRESS_ADD] = assignment_type.Value(ADDRESS_ADD, default=True)
            assignment_type.values[SETTLEMENT_SCHEME] = assignment_type.Value(SETTLEMENT_SCHEME)

        with sdk2.parameters.Group("Binaries") as binaries:
            prepare_buildings_udf = sdk2.parameters.Resource(
                "Sandbox resource ID of prepare_buildings python udf",
                resource_type=MapsPrepareBuildingsForAddressPushesBinary,
                required=False,
            )
            prepare_dwellplaces_executable = sdk2.parameters.Resource(
                "Sandbox resource ID of prepare_dwellpalces executable",
                resource_type=MapsPrepareDwellplacesForAddressPushesBinary,
                required=False,
            )
            make_pushes_udf = sdk2.parameters.Resource(
                "Sandbox resource ID of make_pushes python udf",
                resource_type=MapsMakeAddressPushesBinary,
                required=False,
            )

        with sdk2.parameters.Group("Data settings") as data_settings:
            distance_threshold = sdk2.parameters.Integer(
                "Max distance in meters between user dwellplace and "
                "the point we ask him for",
                default=200,
            )

    class Requirements(sdk2.Requirements):
        environments = [
            PipEnvironment("yandex-yt")
        ]

    # empty initial context
    class Context(sdk2.Context):
        pass

    def _make_prepare_buildings_subtask(self):
        return MapsExecRunPythonUdf(
            self,
            binary=self.Parameters.run_python_udf_executable,
            subbinary=self.Parameters.prepare_buildings_udf,
            env_vars=self.Context.env_vars,
            args="--output {}".format(
                self.Context.buildings_path
            ),
            kill_timeout=3600,  # 1 hour
            description="Prerpare buildings for address pushes",
            notifications=self.Parameters.notifications,
        ).enqueue()

    def _make_prepare_dwellplaces_subtask(self):
        return MapsExecTask(
            self,
            binary=self.Parameters.prepare_dwellplaces_executable,
            env_vars=self.Context.env_vars,
            args="--output-table {} --start-date {} --end-date {}".format(
                self.Context.dwells_path,
                self.Context.start_date,
                self.Context.end_date
            ),
            kill_timeout=2 * 3600,  # 2 hours
            description="Prerpare dwellplaces for address pushes",
            notifications=self.Parameters.notifications,
        ).enqueue()

    def _make_make_pushes_subtask(self):
        return MapsExecRunPythonUdf(
            self,
            binary=self.Parameters.run_python_udf_executable,
            subbinary=self.Parameters.make_pushes_udf,
            env_vars=self.Context.env_vars,
            args=(
                "--output {} "
                "--dwells {} "
                "--buildings {} "
                "--uuids {} "
                "--distance-threshold {} "
                "--assignment-type {}"
            ).format(
                self.Context.raw_pushes_path,
                self.Context.dwells_path,
                self.Context.buildings_path,
                UUIDS_PATH_TEMPLATE.format(
                    start_date=self.Context.start_date,
                    end_date=self.Context.end_date
                ),
                self.Parameters.distance_threshold,
                self.Parameters.assignment_type,
            ),
            kill_timeout=8 * 3600,  # 8 hours
            description="Prepare {} pushes".format(self.Parameters.assignment_type),
            notifications=self.Parameters.notifications,
        ).enqueue()

    def _set_context(self):
        import yt.wrapper as yt
        yt.config.set_proxy("hahn")
        yt.config["token"] = get_vault_data(self.Parameters.yt_token)
        self.Context.env_vars = {
            "YT_TOKEN": self.Parameters.yt_token,
            "YQL_TOKEN": self.Parameters.yql_token,
        }

        today = datetime.date.today()
        self.Context.end_date = (
            self.Parameters.end_date
            if not self.Parameters.use_last_date
            else (today - datetime.timedelta(days=1)).isoformat()
        )
        self.Context.start_date = (
            datetime.datetime.strptime(self.Context.end_date, "%Y-%m-%d") -
                datetime.timedelta(days=self.Parameters.n_days)).date().isoformat()

        self.Context.folder = yt.ypath_join(self.Parameters.yt_path, today.isoformat())
        if not yt.exists(self.Context.folder):
            yt.create("map_node", self.Context.folder)

        self.Context.buildings_path = yt.ypath_join(self.Context.folder, "buildings")
        self.Context.dwells_path = yt.ypath_join(self.Context.folder, "dwells")
        self.Context.raw_pushes_path = yt.ypath_join(self.Context.folder, "raw_pushes")

    def on_enqueue(self):
        super(MakeAddressAndSettlementsAssignmentsPushes, self).on_enqueue()
        ensure_latest_resources(self)

    def on_execute(self):
        check_subtasks(self)
        with self.memoize_stage.set_inputs:
            self._set_context()

        with self.memoize_stage.prepare_data:
            raise wait_tasks([
                self._make_prepare_buildings_subtask(),
                self._make_prepare_dwellplaces_subtask()
            ])
            check_subtasks(self)

        with self.memoize_stage.make_pushes:
            raise wait_tasks([
                self._make_make_pushes_subtask()
            ])
            check_subtasks(self)
