#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
from sandbox import sdk2
from sandbox.projects.nmaps.ugc_assignments.common.resources import (
    MapsAbstractBinaryForPushes,
)
from sandbox.projects.nmaps.ugc_assignments.common.task_utils import (
    check_subtasks,
    ensure_latest_resources,
    get_vault_data,
    wait_tasks,
)
from sandbox.projects.nmaps.ugc_assignments.common.ugc_pushes_parameters import (
    BuildAssignmentsParameters,
)
from sandbox.projects.nmaps.ugc_assignments.ExecRunPythonUdf import (
    MapsExecRunPythonUdf,
)
from sandbox.sandboxsdk.environments import PipEnvironment


class MapsPrepareEntrancesForUgcEntrancesAssignments(MapsAbstractBinaryForPushes):
    pass


class MapsPrepareUsersForUgcEntrancesAssignments(MapsAbstractBinaryForPushes):
    pass


class MapsJoinEntrancesWithUsersForUgcEntrancesAssignments(MapsAbstractBinaryForPushes):
    pass


UUIDS_PATH_TEMPLATE = "//home/maps-nmaps/analytics/uuids_for_pushes/{{{start_date}..{end_date}}}"
DWELLS_PATH_TEMPLATE = "//home/user_identification/orgvisits/prod/state/dwellplaces/current/{{{start_date}..{end_date}}}"


class MakeEntrancesPushes(sdk2.Task):
    class Parameters(BuildAssignmentsParameters):
        with sdk2.parameters.Group("Binaries") as entrances_binaries:
            prepare_entrances_udf = sdk2.parameters.Resource(
                "Sandbox resource ID of prepare_entrances python udf",
                resource_type=MapsPrepareEntrancesForUgcEntrancesAssignments,
                required=False,
            )
            prepare_users_udf = sdk2.parameters.Resource(
                "Sandbox resource ID of prepare_users python udf",
                resource_type=MapsPrepareUsersForUgcEntrancesAssignments,
                required=False,
            )
            join_users_with_entrances_pushes_udf = sdk2.parameters.Resource(
                "Sandbox resource ID of join_users_with_entrances python udf",
                resource_type=MapsJoinEntrancesWithUsersForUgcEntrancesAssignments,
                required=False,
            )

    class Context(sdk2.Context):
        pass

    class Requirements(sdk2.Requirements):
        environments = [
            PipEnvironment("yandex-yt")
        ]

    def _make_prepare_subtasks(self):
        return [
            MapsExecRunPythonUdf(
                self,
                binary=self.Parameters.run_python_udf_executable,
                subbinary=self.Parameters.prepare_entrances_udf,
                env_vars=self.Context.env_vars,
                args="--output {} --use-maps-nmaps-yt-pool".format(
                    self.Context.entrances_path,
                ),
                kill_timeout=10800,  # 3 hours
                description="Prepare entrances from ymapsdf",
                notifications=self.Parameters.notifications,
            ).enqueue(),
            MapsExecRunPythonUdf(
                self,
                binary=self.Parameters.run_python_udf_executable,
                subbinary=self.Parameters.prepare_users_udf,
                env_vars=self.Context.env_vars,
                args=(
                    "--output {} "
                    "--uuids {} "
                    "--dwells {} "
                    "--use-maps-nmaps-yt-pool"
                ).format(
                    self.Context.users_path,
                    UUIDS_PATH_TEMPLATE.format(
                        start_date=self.Context.start_date,
                        end_date=self.Context.end_date
                    ),
                    DWELLS_PATH_TEMPLATE.format(
                        start_date=self.Context.start_date,
                        end_date=self.Context.end_date
                    ),
                ),
                kill_timeout=4 * 3600,  # 4 hours
                description="Prerpare users for entrances assignments",
                notifications=self.Parameters.notifications,
            ).enqueue(),
        ]

    def _make_join_subtask(self):
        return MapsExecRunPythonUdf(
            self,
            binary=self.Parameters.run_python_udf_executable,
            subbinary=self.Parameters.join_users_with_entrances_pushes_udf,
            env_vars=self.Context.env_vars,
            args=(
                "--assignments-output {} "
                "--pushes-output {} "
                "--users {} "
                "--entrances {} "
                "--use-maps-nmaps-yt-pool"
            ).format(
                self.Context.ugc_assignments_path,
                self.Context.assignments_path,
                self.Context.users_path,
                self.Context.entrances_path
            ),
            kill_timeout=4 * 3600,  # 4 hours
            description="Prepare entrances pushes",
            notifications=self.Parameters.notifications,
        ).enqueue()

    def _set_context(self):
        import yt.wrapper as yt
        yt.config.set_proxy("hahn")
        yt.config["token"] = get_vault_data(self.Parameters.yt_token)
        self.Context.env_vars = {
            "YT_TOKEN": self.Parameters.yt_token,
            "YQL_TOKEN": self.Parameters.yql_token,
        }

        today = datetime.date.today()
        self.Context.end_date = (
            self.Parameters.end_date
            if not self.Parameters.use_last_date
            else (today - datetime.timedelta(days=1)).isoformat()
        )
        self.Context.start_date = (
            datetime.datetime.strptime(self.Context.end_date, "%Y-%m-%d") -
                datetime.timedelta(days=self.Parameters.n_days)).date().isoformat()

        self.Context.folder = yt.ypath_join(self.Parameters.yt_path, today.isoformat())
        if not yt.exists(self.Context.folder):
            yt.create("map_node", self.Context.folder)

        self.Context.entrances_path = yt.ypath_join(self.Context.folder, "entrances")
        self.Context.users_path = yt.ypath_join(self.Context.folder, "users")
        self.Context.ugc_assignments_path = yt.ypath_join(self.Context.folder, self.Parameters.ugc_assignments)
        self.Context.assignments_path = yt.ypath_join(self.Context.folder, self.Parameters.assignments)

    def on_enqueue(self):
        super(MakeEntrancesPushes, self).on_enqueue()
        ensure_latest_resources(self)

    def on_execute(self):
        check_subtasks(self)
        with self.memoize_stage.set_inputs:
            self._set_context()

        with self.memoize_stage.prepare_data:
            prepare_subtasks = self._make_prepare_subtasks()
            raise wait_tasks(prepare_subtasks)
            check_subtasks(self)

        with self.memoize_stage.join_users_with_entrances:
            subtask = self._make_join_subtask()
            raise wait_tasks([subtask])
            check_subtasks(self)
