#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.nmaps.ugc_assignments.common.resources import (
    MapsAbstractBinaryForPushes,
)
from sandbox.projects.nmaps.ugc_assignments.common.task_utils import (
    check_subtasks,
    get_vault_data,
    ensure_latest_resources,
    wait_tasks,
)
from sandbox.projects.nmaps.ugc_assignments.PrepareUgcPushes.prepare_pushes import (
    make_create_ugc_tasks_subtask,
    make_filter_pushes_subtask,
    make_pack_pushes_subtask,
    make_send_pushes_subtask,
)
from sandbox.projects.nmaps.ugc_assignments.common.ugc_pushes_parameters import (
    UgcPushesParameters,
)
from sandbox.sandboxsdk.environments import PipEnvironment


SUP_LOGS_TABLE = "sup_logs"
TVM_SECRET_YAV_ID_TESTING = "sec-01eynrac9jcfqwjqkvh6vh8c34"
TVM_SECRET_YAV_ID_STABLE = "sec-01eynrfrq9j0yd556fxbgzqeak"


# same values as parameters in create_ugc_account_tasks script
# https://a.yandex-team.ru/arc/trunk/arcadia/maps/poi/notification/bin/create_ugc_account_tasks/main.py?rev=r8268916#L56
ORG_STATUS = "org_status"
ADDRESS_ADD = "address_add"
BARRIER_EDIT = "barriers"
GATE_EDIT = "gates"
ENTRANCE_EDIT = "entrances"
SETTLEMENT_SCHEME = "settlement_scheme"

PUSH_TYPES = {
    ORG_STATUS: "geoplatform_org_status",
    ADDRESS_ADD: "geoplatform_address_request",
    BARRIER_EDIT: "geoplatform_barrier_edit",
    GATE_EDIT: "geoplatform_gate_edit",
    ENTRANCE_EDIT: "geoplatform_entrances_edit",
    SETTLEMENT_SCHEME: "geoplatform_settlement_scheme",
}


class MapsSendPushesBinary(MapsAbstractBinaryForPushes):
    pass


class MapsPackPushesBinary(MapsAbstractBinaryForPushes):
    pass


class MapsFilterPushesBinary(MapsAbstractBinaryForPushes):
    pass


class CreateUgcAccountTasksBinary(MapsAbstractBinaryForPushes):
    pass


class MapsPrepareUgcPushesTask(sdk2.Task):
    class Parameters(UgcPushesParameters):
        with sdk2.parameters.RadioGroup("Assignment type") as assignment_type:
            assignment_type.values[ADDRESS_ADD] = assignment_type.Value(ADDRESS_ADD, default=True)
            assignment_type.values[BARRIER_EDIT] = assignment_type.Value(BARRIER_EDIT)
            assignment_type.values[GATE_EDIT] = assignment_type.Value(GATE_EDIT)
            assignment_type.values[ENTRANCE_EDIT] = assignment_type.Value(ENTRANCE_EDIT)
            assignment_type.values[ORG_STATUS] = assignment_type.Value(ORG_STATUS)
            assignment_type.values[SETTLEMENT_SCHEME] = assignment_type.Value(SETTLEMENT_SCHEME)

        with sdk2.parameters.Group("YT params") as make_pushes_yt_params:
            yt_subdir = sdk2.parameters.String(
                "Subfolder to store results, last available date in folder by default",
                required=False,
            )
            assignments = sdk2.parameters.String(
                "YT-table name for output from create_assignments and input for filter_pushes",
                default="assignments",
            )
            sup_pushes = sdk2.parameters.String(
                "YT-table name with sup format pushes table",
                default="sup_pushes",
            )

        with sdk2.parameters.Group("Stage parameters") as stage_parameters:
            create_assignments_flag = sdk2.parameters.Bool("Create assignments in ugc account")
            with create_assignments_flag.value[True]:
                with sdk2.parameters.Group("YT tables names") as ugc_yt_tables_names:
                    ugc_assignments = sdk2.parameters.String(
                        "YT-table name with assignments to send to ugc account",
                        default="ugc_assignments",
                    )
                create_ugc_account_tasks_binary = sdk2.parameters.Resource(
                    "Sandbox resource ID of create_ugc_account_tasks executable",
                    resource_type=CreateUgcAccountTasksBinary,
                    required=False,
                )
                create_tasks_jobs = sdk2.parameters.Integer(
                    "Number of jobs creating assignments in parallel",
                    default=16
                )

            prepare_pushes_flag = sdk2.parameters.Bool("Filter & pack pushes")
            with prepare_pushes_flag.value[True]:

                with sdk2.parameters.Group("YT tables names") as pushes_yt_tables_names:
                    pushes = sdk2.parameters.String(
                        "YT-table name with filtered pushes table",
                        default="pushes",
                    )
                with sdk2.parameters.Group("Binaries") as pack_pushes_binaries:
                    filter_pushes_udf = sdk2.parameters.Resource(
                        "Sandbox resource ID of filter_pushes python udf",
                        resource_type=MapsFilterPushesBinary,
                        required=False,
                    )
                    pack_pushes_binary = sdk2.parameters.Resource(
                        "Sandbox resource ID of send_pushes executable",
                        resource_type=MapsPackPushesBinary,
                        required=False,
                    )
                    cooldown = sdk2.parameters.Integer(
                        "Cooldown, days: do not send push for user if user marked "
                        "such push as useless or opened the same one during cooldown days",
                        default=60
                    )

            send_pushes_flag = sdk2.parameters.Bool("Send pushes")
            with send_pushes_flag.value[True]:
                logbroker_vault = sdk2.parameters.Vault("Logbroker vault", required=True)
                logbroker_username = sdk2.parameters.String("Logbroker username", required=True)

                send_pushes_binary = sdk2.parameters.Resource(
                    "Sandbox resource ID of send_pushes executable",
                    resource_type=MapsSendPushesBinary,
                    required=False,
                )

        with sdk2.parameters.RadioGroup("Environment") as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = environment.Value("stable")

    class Requirements(sdk2.Requirements):
        environments = [
            PipEnvironment("yandex-yt")
        ]

    # empty initial context
    class Context(sdk2.Context):
        pass

    def _get_last_pushes_dir(self):
        import yt.wrapper as yt
        for date_dir in sorted(yt.list(self.Parameters.yt_path, absolute=False), reverse=True):
            sup_logs = yt.ypath_join(self.Parameters.yt_path, date_dir, SUP_LOGS_TABLE)
            if yt.exists(sup_logs):  # last pack of pushes is already sent
                self.set_info("All pushes are sent for {}".format(date_dir))
                return None
            else:  # prepared but not sent
                return date_dir
        return None

    def _set_context(self):
        import yt.wrapper as yt
        yt.config.set_proxy("hahn")
        yt.config["token"] = get_vault_data(self.Parameters.yt_token)
        self.Context.env_vars = {
            "YT_TOKEN": self.Parameters.yt_token,
            "YQL_TOKEN": self.Parameters.yql_token,
        }

        subdir = self.Parameters.yt_subdir or self._get_last_pushes_dir()
        if subdir is None:
            raise TaskError("Nothing to send in {}".format(self.Parameters.yt_path))

        self.Context.folder = yt.ypath_join(self.Parameters.yt_path, subdir)
        if not yt.exists(self.Context.folder):
            yt.create("map_node", self.Context.folder)

        self.Context.assignments_path = yt.ypath_join(self.Context.folder, self.Parameters.assignments)
        self.Context.sup_pushes_path = yt.ypath_join(self.Context.folder, self.Parameters.sup_pushes)

        if self.Parameters.create_assignments_flag:
            tvm_secret_id = TVM_SECRET_YAV_ID_TESTING
            if self.Parameters.environment == "stable":
                tvm_secret_id = TVM_SECRET_YAV_ID_STABLE

            self.Context.tvm_secret = sdk2.yav.Secret(tvm_secret_id).data()["client_secret"]
            self.Context.ugc_assignments_path = yt.ypath_join(self.Context.folder, self.Parameters.ugc_assignments)

        if self.Parameters.prepare_pushes_flag:
            self.Context.today = (datetime.date.today() - datetime.timedelta(days=1)).isoformat()
            self.Context.pushes_path = yt.ypath_join(self.Context.folder, self.Parameters.pushes)
            try:
                self.Context.push_type = PUSH_TYPES[self.Parameters.assignment_type]
            except KeyError:
                raise TaskError(
                    "Unknown assignment type '{}'".format(self.Parameters.assignment_type)
                )

        if self.Parameters.send_pushes_flag:
            self.Context.sup_logs_path = yt.ypath_join(self.Context.folder, SUP_LOGS_TABLE)

    def on_enqueue(self):
        super(MapsPrepareUgcPushesTask, self).on_enqueue()
        ensure_latest_resources(self)

    def on_execute(self):
        check_subtasks(self)
        with self.memoize_stage.set_inputs:
            self._set_context()

        if self.Parameters.create_assignments_flag:
            with self.memoize_stage.create_assignments:
                subtask = make_create_ugc_tasks_subtask(self)
                raise wait_tasks([subtask])
                check_subtasks(self)

        if self.Parameters.prepare_pushes_flag:
            with self.memoize_stage.filter_pushes:
                subtask = make_filter_pushes_subtask(self)
                raise wait_tasks([subtask])
                check_subtasks(self)
            with self.memoize_stage.pack_pushes:
                subtask = make_pack_pushes_subtask(self)
                raise wait_tasks([subtask])
                check_subtasks(self)

        if self.Parameters.send_pushes_flag:
            with self.memoize_stage.send_pushes:
                subtask = make_send_pushes_subtask(self)
                raise wait_tasks([subtask])
                check_subtasks(self)
