import logging

from sandbox import sdk2
from sandbox.sdk2 import environments


class WmsSupportTicket(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("WMS database settings", collapse=True) as wms_db_settings:
            wms_db_host = sdk2.parameters.String('WMS database host', required=True)
            wms_db_port = sdk2.parameters.String('WMS database port', required=True)
            wms_db_name = sdk2.parameters.String('WMS database name', required=True)
            wms_db_user_secret = sdk2.parameters.YavSecret(
                'WMS database username'
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: username',
                required=True
            )
            wms_db_password_secret = sdk2.parameters.YavSecret(
                'WMS database password '
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: password',
                required=True
            )

        with sdk2.parameters.Group("Startrack settings", collapse=True) as st_settings:
            st_user_agent = sdk2.parameters.String('Startrack user agent', required=True)
            st_url = sdk2.parameters.String('Startrack url', required=True)
            st_token_secret = sdk2.parameters.YavSecret(
                'Startrack token'
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: token',
                required=True
            )
            st_queue = sdk2.parameters.String('Startrack queue', required=True)
            st_summary = sdk2.parameters.String('Startrack ticket summary', required=True)
            st_tags = sdk2.parameters.List('List of tags for Startrek issue')
            board = sdk2.parameters.String('Board URL')

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (
            environments.PipEnvironment('pymssql==2.1.4'),
            environments.PipEnvironment('yandex_tracker_client', version="1.3",
                                        custom_parameters=["--upgrade-strategy only-if-needed"]),
            environments.PipEnvironment('startrek_client', version="2.3.0",
                                        custom_parameters=["--upgrade-strategy only-if-needed"]),
        )

    def request_wms(self, user, password):
        conn = None
        cur = None
        try:
            logging.info('Connecting to the WMS database...')
            logging.info('User: {}'.format(user))
            import pymssql
            with pymssql.connect(
                user=user,
                password=password,
                host=self.Parameters.wms_db_host,
                port=self.Parameters.wms_db_port,
                database=self.Parameters.wms_db_name
            ) as conn:
                logging.info('Connected')

                with conn.cursor() as cur:
                    cur.execute(
                        """
                            select lli.storerkey as [storerkey], lli.sku as [sku], lli.lot as [lot], lli.loc as [loc],
                            lli.qty as [qty], lli.id as [id],
                            isnull(lli.qtyallocated, 0) as [lli_qtyallocated], isnull(lli.qtypicked, 0)
                            as [lli_qtypicked],
                            isnull(pda.qty, 0) as [pd_qtyallocated], isnull(pdp.qty, 0) as [pd_qtypicked]
                            from scprd.wmwhse1.lotxlocxid lli
                            full outer join (select storerkey, sku, lot, loc, id, sum(qty) as qty
                            from scprd.wmwhse1.pickdetail pd
                            where status between '0' and '4' group by storerkey, sku, lot, loc, id) pda
                            on lli.storerkey = pda.storerkey and lli.lot = pda.lot and lli.loc = pda.loc
                            and lli.id = pda.id
                            full outer join (select storerkey, sku, lot, loc, id, sum(qty) as qty
                            from scprd.wmwhse1.pickdetail pd
                            where status between '5' and '8' group by storerkey, sku, lot, loc, id) pdp
                            on lli.storerkey = pdp.storerkey and lli.lot = pdp.lot and lli.loc = pdp.loc
                            and lli.id = pdp.id
                            where ( isnull(lli.qtyallocated, 0) <> isnull(pda.qty, 0)
                            OR isnull(lli.qtypicked, 0) <> isnull(pdp.qty, 0))
                        """
                    )
                    data = ''
                    data = cur.fetchall()
        except Exception as error:
            logging.info(error)
        finally:
            logging.info('WMS database connection closed.')
        return data

    def create_startrek_ticket(self, summary, description, st_token):
        from startrek_client import Startrek
        st = Startrek(
            useragent=self.Parameters.st_user_agent,
            base_url=self.Parameters.st_url,
            token=st_token
        )

        st.issues.create(
            queue=self.Parameters.st_queue,
            summary=self.Parameters.st_summary + ' - ' + summary,
            type={'name': 'Technical Task'},
            description=description,
            tags=self.Parameters.st_tags
        )

    def on_execute(self):
        wms_db_user = self.Parameters.wms_db_user_secret.data()
        wms_db_user_key = self.Parameters.wms_db_user_secret.default_key or 'username'
        wms_db_user = wms_db_user[wms_db_user_key]
        wms_db_password = self.Parameters.wms_db_password_secret.data()
        wms_db_password_key = self.Parameters.wms_db_password_secret.default_key or 'password'
        wms_db_password = wms_db_password[wms_db_password_key]
        st_token = self.Parameters.st_token_secret.data()
        st_token_key = self.Parameters.st_token_secret.default_key or 'token'
        st_token = st_token[st_token_key]

        data = self.request_wms(wms_db_user, wms_db_password)
        ticket_description = '| storerkey | sku | lot | loc | qty | id | lli_qtyallocated | lli_qtypicked |' \
                             ' pd_qtyallocated | pd_qtypicked ' \
                             '|\n| ---: | ---: | ---: | ---: | ---: | ---: | ---: | ---: | ---: | ---: |'
        for storerkey, sku, lot, loc, qty, id, lli_qtyallocated, lli_qtypicked, pd_qtyallocated, pd_qtypicked in data:
            ticket_description = ticket_description + '\n| {} | {} | {} | {} | {} | {} | {} | {} | {} | {} |'.format(
                storerkey, sku, lot, loc, qty, id, lli_qtyallocated, lli_qtypicked, pd_qtyallocated, pd_qtypicked)
        self.create_startrek_ticket('Test ticket', ticket_description, st_token)
