# coding=utf-8

from datetime import datetime as dd
import logging
import platform
import time
import uuid

import pandas


# HACK TODO!  Delete it after ABC ids are added to Matilda
# NOCDEV-2647
from sandbox.projects.noc.ExportNetworkMetering.abc import services
from sandbox.projects.noc.ExportNetworkMetering.peers import get_peers
import sandbox.projects.noc.ExportNetworkMetering.util as util

SKU_DCI_RU = 'net.bandwidth.dci.RU-CENTRAL'
SKU_DCI_FI = 'net.bandwidth.dci.FIN'
SKU_EXT = 'net.bandwidth.ext'


logger = logging.getLogger(__name__)

NOC_ABC_ID = 922

DC_LIST = ('VLA', 'MAN', 'SAS', 'MYT', 'IVA')
FOREIGN_ROUTERS = ('ams1-b1', 'ash-b1', 'fra1-b1', 'jansson', 'sibelius')

UNKNOWN_MARKERS = ('__root__', '(unknown)')

HOSTNAME = platform.node()


# AS per https://st.yandex-team.ru/CLOUD-46466
def format_interdc(data, hour):
    res = []
    unknowns = set()
    for datum in data:
        # Two records for each interaction.
        for direct, reverse, direction in (
                ('src', 'dst', 'egress'),
                ('dst', 'src', 'ingress')):
            down, rown = datum[direct + '_own'], datum[reverse + '_own']
            dmacro, rmacro = datum[direct + '_macro'], datum[reverse + '_macro']
            if '_total_' in (down, rown, dmacro, rmacro):
                continue
            src_srv = services.get(down)
            dst_srv = services.get(rown)
            if None in (src_srv, dst_srv):
                if down in UNKNOWN_MARKERS:
                    unknowns.add(down)
                if rown in UNKNOWN_MARKERS:
                    unknowns.add(rown)
                continue

            dc = datum[direct + '_dc']
            if dc not in DC_LIST:
                continue

            res.append(dict(
                id=str(uuid.uuid1()),
                abc_id=src_srv,
                source_wt=int(time.time()),
                source_id=HOSTNAME,
                schema='net.generic.4',
                usage={
                    'start': hour,
                    'finish': hour + 3600,
                    'type': 'delta',
                    'quantity': datum['traf'],
                    'unit': 'byte'
                },
                tags={
                    'type': 'interdc',
                    'subtype': 'international' if dc == 'MAN' else 'domestic',
                    'direction': direction,
                    'src_dc': dc,
                    'dst_dc': dc,
                    'src_service': src_srv,
                    'dst_service': dst_srv,
                    'src_macro': dmacro,
                    'dst_macro': rmacro
                },
            ))
    for unknown in unknowns:
        logger.debug("Unknown service", unknown)
    logger.info("%i messages formatted out of %i items", len(res), len(data))
    return res


def format_external(data, hour):
    res = []
    unknowns = set()
    for datum in data:
        if '_total_' in (datum["src_own"], datum["dst_own"],
                         datum["src_macro"], datum["dst_macro"]):
            continue
        src_srv = services.get(datum["src_own"])
        if src_srv == "(unknown)":
            src_srv = services.get(datum["dst_own"])
        if src_srv is None:
            unknowns.add(datum["src_own"])
            continue

        if "agent_name" in datum:
            peer = get_peers().get((datum["agent_name"], datum["iface_name"]))
            if not peer:
                unknowns.add("%s@%s" % (datum["agent_name"], datum["iface_name"]))
                continue

        res.append(dict(
            id=str(uuid.uuid1()),
            abc_id=src_srv,
            source_wt=int(time.time()),
            source_id=HOSTNAME,
            schema='net.generic.4',
            usage={
                'start': hour,
                'finish': hour + 3600,
                'type': 'delta',
                'quantity': datum['traf'],
                'unit': 'byte'
            },
            tags={
                'type': 'external',
                'subtype': datum["iface"],
                'name': datum["iface"],
                'src_service': src_srv,
                'src_macro': datum['src_macro'],
                'direction': 'egress',
            },
        ))
    for unknown in unknowns:
        logger.info("Unknown peering record for %s" % unknown)
    logger.info(
        "%i messages formatted out of %i items; totals count is %i",
        len(res), len(data), len(unknowns))
    return res


def format_interdc_totals(data, hour):
    records = []
    for host, ifaces in data.items():
        for iface, dirs in ifaces.items():
            records.append((host, "rx", iface, dirs["rx"]))
            records.append((host, "tx", iface, dirs["tx"]))
    df = pandas.DataFrame(records, columns=["host", "dir", "iface", "bytes"])

    res = []
    for host, dir_, bytes in df.groupby(["host", "dir"]).sum().to_records():
        dc = host[0:3].upper()
        if dc not in DC_LIST:
            continue

        res.append(dict(
            id=str(uuid.uuid1()),
            source_wt=int(time.time()),
            source_id=HOSTNAME,
            schema='net.totals.2',
            abc_id=NOC_ABC_ID,
            usage={
                'start': hour,
                'finish': hour + 3600,
                'type': 'delta',
                'quantity': bytes,
                'unit': 'byte'
            },
            tags={
                'type': 'interdc',
                'subtype': 'international' if dc == "MAN" else "domestic",
                'name': dc,
                'direction': "egress" if dir_ == "tx" else "ingress",
            },
        ))
    return res


def format_external_totals(data, hour):
    records = []
    for host, ifaces in data.items():
        for iface, dirs in ifaces.items():
            records.append((host, "rx", iface, dirs["rx"]))
            records.append((host, "tx", iface, dirs["tx"]))
    df = pandas.DataFrame(records, columns=["host", "dir", "iface", "bytes"])

    res = []
    for host, dir_, bytes in df.groupby(["host", "dir"]).sum().to_records():
        res.append(dict(
            id=str(uuid.uuid1()),
            source_wt=int(time.time()),
            source_id=HOSTNAME,
            schema='net.totals.2',
            abc_id=NOC_ABC_ID,
            usage={
                'start': hour,
                'finish': hour + 3600,
                'type': 'delta',
                'quantity': bytes,
                'unit': 'byte'
            },
            tags={
                'type': 'external',
                'subtype': 'foreign' if host in FOREIGN_ROUTERS else "domestic",
                'name': host,
                'direction': "egress" if dir_ == "tx" else "ingress",
            },
        ))
    return res


def format_p95(data, sku, month, next_month):
    res = []
    for datum in data:
        res.append({
            "resource_id": "foodqkuj4ge5j9sg34uq",
            "tags": {
                # Это сам SKU, тут ничего не надо вычислять.
                "type": sku,
                # А вот эти поля, увы, в агрегатных данных бессмысленно представлять,
                # потому что они при арифметическом суммировании размеченных ими данных
                # не дают данных по потреблению (дают искажённые данные):
                # direction
                # src/dst_dc
                # src/dst_service
                # src/dst_macro
                # src/dst_project_id
            },
            "id": str(uuid.uuid1()),
            # На кого побиллили
            "abc_id": services.get(datum["owner"]),
            "usage": {
                # Начало месяца
                "start": util.to_timestamp(month),
                # Текущий час
                "finish": min(util.to_timestamp(next_month), util.to_timestamp(dd.utcnow())),
                "type": "cumulative",
                # Текущее значение израсходованной пропускной полосы по 95-ой процентили;
                # первые 5% месяца будет равно нулю, а после этого минимальному замеру из топ-5% замеров
                "quantity": int(datum["p95"]),
                "unit": "bit/second"
            },
            # узел-отправитель
            "source_id": HOSTNAME,
            "schema": "net.generic.4",
            "source_wt": int(time.time())
        })
    return res
