# coding=utf-8
import logging

import json
import time

import kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api as pqlib
import kikimr.public.sdk.python.persqueue.errors as pqerrors


logger = logging.getLogger(__name__)


class LogBroker:
    def __init__(
            self, cred_provider,
            endpoint, port, topic, source_id,
            throttle_pause=0):
        logger.info("Spawning a Logbroker client %s", source_id)
        # retrying_producer spams full message contents by default
        for boring_logger, level in (
            ('kikimr.public.sdk.python.persqueue._core',
             logging.INFO),
            ('ydb.connection',
             logging.INFO),
            ('ydb.resolver.DiscoveryEndpointsResolver',
             logging.WARN)):
            logging.getLogger(boring_logger).setLevel(level)

        self._source_id = source_id
        self._throttle_pause = throttle_pause

        logger.debug('Logbroker connector %s instance to %s', source_id, endpoint)
        self.api = pqlib.PQStreamingAPI(endpoint, port)
        api_start_future = self.api.start()
        result = api_start_future.result(timeout=30)
        logger.debug('Logbroker API %s started: %s', source_id, result)

        configurator = pqlib.ProducerConfigurator(topic, source_id)
        self.producer = self.api.create_retrying_producer(
            configurator, credentials_provider=cred_provider)

        logger.debug('Starting Logbroker producer %s', source_id)
        start_future = self.producer.start()
        start_result = start_future.result(timeout=10)
        if (isinstance(start_result, pqerrors.SessionFailureResult)
            or not start_result.HasField('init')):
            raise RuntimeError(
                'Logbroker producer %s (%s @ %s) start failure: %s' % (
                    source_id, topic, endpoint, start_result))
        self.max_seq = start_result.init.max_seq_no + 1
        logger.debug(
            'Logbroker producer %s started: %s', source_id, start_result)

    def send(self, messages, start_seq=None):
        if not messages:
            logger.info('No messages to be sent by %s.' % self._source_id)
            return
        seq = start_seq or self.max_seq
        sid = self._source_id
        logger.debug('Start %s seq: %s', sid, seq)

        write_responses = []
        for message in messages:
            # write() requires seq_no and message itself and returns a
            # Future() object result will be set when server confirms
            # message written. Result will contain Ack (see protos for
            # more details)
            response = self.producer.write(seq, json.dumps(message))
            # There is no must to wait for every single ack, we may go
            # further with writing data and wait confirmation later
            write_responses.append(response)
            # Every next message within one source id must be provided
            # with increasing SeqNo.
            seq += 1
            if seq % 1000 == 0:
                logger.debug('%s sending, seq = %i', sid, seq)
                logger.debug(json.dumps(message))
                time.sleep(self._throttle_pause)
        logger.debug('%s messages enqueued by %s.', len(write_responses), sid)

        acks_count = 0
        for r in write_responses:
            write_result = r.result(timeout=10)
            if write_result.HasField('ack'):
                acks_count += 1

        logger.debug('%s: %s messages confirmed out of %i (%i%%).' % (
            sid, acks_count, len(messages), 100 * acks_count // len(messages)))
        logger.debug('%s done sending messages', sid)
        if acks_count < len(messages):
            raise RuntimeError(
                'Some messages have not been sent by %s (see above)!' % sid)

    def stop(self):
        self.producer.stop()
