"""

"""
from collections import defaultdict
import datetime
import json
import logging

try:
    import http.client as httplib
except ImportError:
    import httplib


logger = logging.getLogger(__name__)


CONTENT_TYPE = 'text/plain'
RESPONSE_TYPE = 'application/json'

SOLOMON_HOST = 'solomon.yandex.net'
# time_from and time_to are ISO8601 with milliseconds like 2020-11-06T12:00:00.000Z
SOLOMON_URL_TEMPLATE = (
    '/api/v2/projects/noc/sensors/data?maxPoints=500&from=%(time_from)s&to=%(time_to)s'
)


class Solomon(object):
    def __init__(self, token):
        self._token = token

    @staticmethod
    def make_request(host, ifnames):
        return (
            'sum({cluster="all", host="%(host)s", ifname="%(ifnames)s", '
            'service="network", sensor="rx|tx"}) by 1h'
        ) % {"host": host, "ifnames": "|".join(ifnames)}

    def get_data(self, requestees, time_from, time_to):
        """
        Fetches data from Solomon pertaining to all hosts and interfaces
        found in the input parameter.  Then it will filter data by interface
        and switch name according to the mapping.  Then it will return

        :param requestees: a `dict` where keys are device names and
            values are `list`s of interfaces.

        :param time_from: `datetime.datetime`
        :param time_to: `datetime.datetime`

        :return: {switch_name: [{interface_name: {"rx|tx": bytes}]}
        """
        res = defaultdict(lambda: defaultdict(lambda: {}))

        request_params = {
            "time_from": time_from.isoformat() + ".000Z",
            "time_to": time_to.isoformat() + ".000Z"
        }
        conn = httplib.HTTPSConnection(SOLOMON_HOST, 443, timeout=30)

        for hostname, ifnames in requestees.items():
            headers = {
                "Authorization": "OAuth %s" % self._token,
                "Content-type": CONTENT_TYPE,
                "Accept": RESPONSE_TYPE,
            }
            conn.request(
                "POST", SOLOMON_URL_TEMPLATE % request_params,
                headers=headers,
                body=self.make_request(hostname, ifnames))
            resp = conn.getresponse(buffering=True)
            if resp.status >= 400:
                raise Exception(
                    "Error contacting Solomon: %s %s %s" % (
                        resp.status, resp.reason, resp.msg))
            data = json.loads(resp.read().decode("utf-8"))

            for ts in data["vector"]:
                datum = ts["timeseries"]
                host = datum["labels"]["host"]
                ifname = datum["labels"]["ifname"]
                sensor = datum["labels"]["sensor"]
                if datum["values"]:
                    bytes_ = int(datum["values"][0])
                    res[host][ifname][sensor] = bytes_

        return res
