# coding=utf-8
import datetime
import json
import logging
import time

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


logger = logging.getLogger(__name__)

# Delete dumps after 1 week
CACHE_TTL = 7
CACHE_ROOT_DIR = 'api_cache'
CACHE_DESCRIPTION = 'API cache for NOC metering etc'

# Cache TTL in ideal circumstances
ABC_SERVICES_CACHE_TTL = 3600
BOT_SERVER_OWNERSHIP_CACHE_TTL = 86400
RT_RES_OWNERSHIP_CACHE_TTL = 3600
WALLE_PROJECT_CACHE_TTL = 3600

# Redo cache if its TTL is below 300 seconds
TTL_TOLERANCE = 300


CACHE_RESOURCE_TYPE = 'NOC_METERING_DATA_CACHE'
MEGABYTE = 2 ** 20


class NocMeteringDataCache(sdk2.Resource):
    """
    Dox https://docs.yandex-team.ru/sandbox/resources
    """
    ttl = CACHE_TTL
    # TODO If the files are accessible separately, can be disabled
    pack_tar = False
    calc_md5 = False
    any_arch = True
    executable = False
    release = 'unstable'


class NocMeteringCacheZapekatel(sdk2.Task):
    """
    Zapechka of caches for NOC metering jobs, Metrafan and whatever
    other droll contraptions there are.
    """

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Caches') as cache_block:
            cache_abc = sdk2.parameters.Bool(
                'Cache ABC service data (id<->slug, slug->[subservice_slug])',
                default_value=True,
                required=True
            )

            cache_bot = sdk2.parameters.Bool(
                'Cache data',
                default_value=True,
                required=True
            )

            cache_rt = sdk2.parameters.Bool(
                'Cache RT abc-objects.php',
                default_value=True,
                required=True
            )

            cache_walle = sdk2.parameters.Bool(
                'Cache hostname -> WallE project (TODO)',
                default_value=True,
                required=True
            )

        with sdk2.parameters.Group('Secrets') as vault_block:
            abc_oauth = sdk2.parameters.Vault(
                'ABC API OAuth vault item', required=True,
                default_value='noc_metering_abc_oauth')

            walle_oauth = sdk2.parameters.Vault(
                'Wall-E API OAuth vault item', required=True,
                default_value='noc_metering_walle_oauth')

    def on_execute(self):
        import sandbox.projects.noc.MeteringCacheZapekatel.fetchdata as fd
        import sandbox.projects.noc.MeteringCacheZapekatel.makecache as mc

        p = self.Parameters

        prev_cache, is_empty = self._try_read_prev_cache()
        prev_data = sdk2.ResourceData(prev_cache)
        cur_cache = self._empty_cache()
        cur_data = sdk2.ResourceData(cur_cache)
        cur_data.path.mkdir(0o755, parents=True, exist_ok=False)
        cur_time = datetime.datetime.utcnow().replace(tzinfo=None)

        for attr, ttl, updater, formatter in [
            ('abc_svc', ABC_SERVICES_CACHE_TTL,
             lambda: fd.fetch_abc_svc(p.abc_oauth.data()), mc.format_abc_svc),
            ('bot_srv', BOT_SERVER_OWNERSHIP_CACHE_TTL,
             fd.fetch_bot_srv, mc.format_bot_srv),
            ('rt_own', RT_RES_OWNERSHIP_CACHE_TTL,
             fd.fetch_rt_own, mc.format_rt_own),
            ('walle_prj', WALLE_PROJECT_CACHE_TTL,
             lambda: fd.fetch_walle_prj(p.walle_oauth), mc.format_walle_prj),
        ]:
            if is_empty or (
                    prev_cache.created.replace(tzinfo=None) + datetime.timedelta(
                        seconds=ttl - TTL_TOLERANCE) < cur_time):
                datum = formatter(updater())
                logger.info('Updated ' + attr)
            else:
                try:
                    datum = json.loads(prev_data.path.joinpath(
                        'cache_' + attr + '.json').read_bytes().decode('utf-8'))
                    logger.info('Kept cached version of ' + attr)
                except Exception, e:
                    logger.exception(e)
                    continue
            cur_data.path.joinpath(
                'cache_' + attr + '.json').write_bytes(
                    json.dumps(datum).encode('utf-8'))

        logger.info('Marking cache as ready.  A new Resource shall be posted.')
        cur_data.ready()

    def _empty_cache(self):
        logger.info("Creating an empty cache instance.")
        return NocMeteringDataCache(self, CACHE_DESCRIPTION, CACHE_ROOT_DIR)

    def _try_read_prev_cache(self):
        from sandbox.agentr.errors import InvalidResource
        try:
            prev_cache = sdk2.Resource.find(
                type=CACHE_RESOURCE_TYPE,
            ).first()
            if not prev_cache:
                logger.info("No caches found.")
                return self._empty_cache(), True
            if prev_cache.state != "READY":
                logger.info(
                    "Cache with state=%s found dating %s; recaching",
                    prev_cache.state, prev_cache.created)
                return self._empty_cache(), True
        except InvalidResource:
            return self._empty_cache(), True
        data = sdk2.ResourceData(prev_cache)
        logger.info(
            "Using binary %s from resource = %s created by %s at %s" % (
                data.path, prev_cache.id, prev_cache.task_id,
                prev_cache.created.isoformat()))
        return prev_cache, False

