#!/usr/bin/env python3

import datetime
import json
import logging
import time

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


logger = logging.getLogger(__name__)

# Delete dumps after 1 year
RESOURCE_TTL = 365
ROOT_DIR = 'top_as'
DESCRIPTION = 'Top ASN by destination traffic'


class NocTopASResource(sdk2.Resource):
    """
    Dox https://docs.yandex-team.ru/sandbox/resources
    """
    ttl = RESOURCE_TTL
    # TODO If the files are accessible separately, can be disabled
    pack_tar = True
    calc_md5 = False
    any_arch = True
    executable = False
    release = 'unstable'


class NocTopAS(sdk2.Task):
    """
    Dump top 200 AS by traffic by ip_ver by direction (thus 800 records total)
    """

    class Parameters(sdk2.Task.Parameters):

        ch_host = sdk2.parameters.String(
            'Clickhouse host',
            default_value='sas-4c2ea4sof0gsxzik.db.yandex.net',
            required=True
        )
        ch_port = sdk2.parameters.Integer(
            'Clickhouse port',
            default_value=9440,
            required=True
        )
        ch_secure = sdk2.parameters.Bool(
            'Secure connection to Clickhouse',
            default_value=True,
            required=True
        )
        ch_db = sdk2.parameters.String(
            'Clickhouse database',
            default_value='tacct_db',
            required=True
        )
        ch_table = sdk2.parameters.String(
            'Clickhouse sflow table (should have src/dst_kind, src/dst_asno, ip_ver)',
            default_value='tacct_sflow',
            required=True
        )

        ch_user = sdk2.parameters.String(
            'Clickhouse user with reading permissions',
            default_value='grafana_ro',
            required=True
        )
        with sdk2.parameters.Group('Secrets') as vault_block:
            ch_password = sdk2.parameters.Vault(
                'Clickhouse password', required=True,
                default_value='metrafan_mdb_ch_password')

    def on_execute(self):
        import sandbox.projects.noc.TopAS.fetchdata as fd

        p = self.Parameters

        resource = NocTopASResource(self, DESCRIPTION, ROOT_DIR)
        res_data = sdk2.ResourceData(resource)
        res_data.path.mkdir(0o755, parents=True, exist_ok=False)

        data = fd.fetch_top_asno_from_ch(fd.Clickhouse(
            server=p.ch_host,
            port=p.ch_port,
            secure=True,
            user=p.ch_user,
            password=p.ch_password.data(),
            database=p.ch_db,
        ))

        resource.path.joinpath('topas200.json').write_bytes(json.dumps(data).encode('utf-8'))
        logger.info('Marking topas200 as ready.  A new Resource shall be posted.')
        res_data.ready()
