"""
Fetch data from CH.
"""

import collections
import datetime

import clickhouse_driver


Clickhouse = collections.namedtuple(
    'Clickhouse', (
        'server',
        'port',
        'secure',

        'user',
        'password',
        'database',
))


TOP_SIZE = 200

TOP_AS_QUERY = """
SELECT
  %(direction)s_asno,
  SUM(bytes) as %(direction)s_bytes,
  SUM(packets) as %(direction)s_packets
FROM tacct_sflow AS d
WHERE
  date = toDate('%(date)s', 'utc')
  AND ip_ver = %(ipver)s
  AND %(direction)s_asno > 0
  AND %(direction)s_kind = 0
GROUP BY %(direction)s_asno
ORDER BY %(direction)s_bytes DESC
LIMIT 200
"""


def fetch_top_asno_from_ch(ch):
    conn = clickhouse_driver.connect(
        host=ch.server, port=ch.port, secure=ch.secure,
        user=ch.user, password=ch.password,
        database=ch.database)

    cursor = conn.cursor()
    top = {"4": {"src": [], "dst": []}, "6": {"src": [], "dst": []}}
    yesterday = str((datetime.datetime.now() - datetime.timedelta(days=1)).date())

    for ipver, direction in (
            (i, d)
            for i in (4, 6)
            for d in ('src', 'dst')):
        cursor.execute(TOP_AS_QUERY % dict(
            date=yesterday,
            ipver=ipver,
            direction=direction
        ))
        while True:
            batch = cursor.fetchmany(TOP_SIZE + 1)
            if not batch:
                break
            top[str(ipver)][direction].extend(
                [{"asno": x[0],
                  "bytes": x[1],
                  "packets": x[2]
                  } for x in batch])
    return {
        "count": TOP_SIZE,
        "ctime": datetime.datetime.utcnow().isoformat() + 'Z',
        "top": top
    }
