from __future__ import unicode_literals

import json
import logging
from datetime import datetime

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.common.types import task as ctt
from sandbox.projects.common.juggler import jclient
import sandbox.projects.common.resource_selectors as resource_selectors

from sandbox.projects.nocdev.cmdb import NOCDEV_CMDB_BINARY, NOCDEV_CMDB_SYNC_FROM_RACKTABLES_DIFF

logger = logging.getLogger(__name__)

DIFF_FILE = "diff.json"


class NocdevCmdbSyncFromRacktables(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 1024
        cores = 1
        ram = 1024
        client_tags = ctc.Tag.MULTISLOT | ctc.Tag.GENERIC

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        cmdb_binary_resource = sdk2.parameters.Resource(
            "CMDB Binary resource",
            resource_type=NOCDEV_CMDB_BINARY,
            multiple=False,
        )
        cmdb_binary_released = sdk2.parameters.String(
            "Or minimal release type of CMDB Binary resource",
            choices=[(t, t) for t in ctt.ReleaseStatus],
            default=ctt.ReleaseStatus.STABLE,
        )

        cmdb_verbose = sdk2.parameters.Bool("CMDB Verbose Logs", default=False)
        diff_resource_ttl = sdk2.parameters.String("TTL (in days) for diff resource", default="14", required=True)

        with sdk2.parameters.Group("Database config"):
            database_url = sdk2.parameters.Url("Database URL", required=True)
            database_password = sdk2.parameters.YavSecret("Database Password", required=True)

        with sdk2.parameters.Group("Racktables config"):
            racktables_url = sdk2.parameters.Url(
                "Racktables Endpoint URL",
                default="https://ro.racktables.yandex-team.ru/api",
                required=True,
            )
            peers_report_url = sdk2.parameters.Url(
                "Peers Report URL",
                default="https://ro.racktables.yandex-team.ru/export/peers-report.php?format=json",
                required=True,
            )
            racktables_token = sdk2.parameters.YavSecret("Racktables Token", required=True)

        with sdk2.parameters.Group("Juggler config"):
            report_to_juggler = sdk2.parameters.Bool("Report to juggler", default=False)
            with report_to_juggler.value[True]:
                juggler_host = sdk2.parameters.String("Juggler host", default="sandbox.NocdevCmdbSyncFromRacktables")
                juggler_service = sdk2.parameters.String("Juggler service", default="task_status")

    def _get_cmdb_binary_resource(self):
        # type: () -> sdk2.Resource
        if self.Parameters.cmdb_binary_resource is not None:
            return self.Parameters.cmdb_binary_resource

        res_ids = []  # type: list[int]
        release_statuses = list(ctt.ReleaseStatus)

        pos = release_statuses.index(self.Parameters.cmdb_binary_released) + 1
        for release_status in release_statuses[:pos]:
            logger.info("Getting last resource with status: %s", release_status)
            res_id, _ = resource_selectors.by_last_released_task(
                NOCDEV_CMDB_BINARY,
                stage=release_status,
            )
            if res_id is not None:
                res_ids.append(res_id)

        if not res_ids:
            raise RuntimeError("No resource with CMDB binary found")

        return sdk2.Resource[max(res_ids)]

    def on_execute(self):
        cmdb_bin_resource = self._get_cmdb_binary_resource()
        logger.info("Downloading resource: %d", cmdb_bin_resource)
        cmdb_bin = sdk2.ResourceData(cmdb_bin_resource)
        logger.info("Finished downloading resource")

        cmdb_cmd = [
            str(cmdb_bin.path), "sync",
            "--database-uri",          self.Parameters.database_url,
            "--diff-path",             DIFF_FILE,
            "--racktables-endpoint",   self.Parameters.racktables_url,
            "--peers-report-endpoint", self.Parameters.peers_report_url,
        ]

        if self.Parameters.cmdb_verbose:
            cmdb_cmd.append("--verbose")

        logger.info("CMDB command: %s", json.dumps(cmdb_cmd))

        with sdk2.helpers.ProcessLog(self, logger="cmdb") as pl:
            sdk2.helpers.subprocess.check_call(
                cmdb_cmd,
                env={
                    "CMDB_TOKEN": self.Parameters.racktables_token.value(),
                    "PGPASSWORD": self.Parameters.database_password.value(),
                },
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

        logger.info("Sync done. Saving diff resource...")

        diff_resource = NOCDEV_CMDB_SYNC_FROM_RACKTABLES_DIFF(
            self,
            "Racktables Sync Diff @ {}".format(datetime.now().isoformat()),
            DIFF_FILE,
            ttl=self.Parameters.diff_resource_ttl,
        )
        diff_resource_data = sdk2.ResourceData(diff_resource)
        diff_resource_data.ready()

    def on_break(self, prev_status, status):
        description = "{task_description}: previous {previous_status} -> current {current_status}".format(
            task_description=self.juggler_task_description, previous_status=prev_status, current_status=status
        )
        self.send_status_to_juggler(status="CRIT", description=description)
        super(NocdevCmdbSyncFromRacktables, self).on_break(prev_status, status)

    def on_success(self, prev_status):
        self.send_status_to_juggler(status="OK")
        super(NocdevCmdbSyncFromRacktables, self).on_success(prev_status)

    def on_failure(self, prev_status):
        description = "{task_description}: previous {previous_status} -> current FAILURE".format(
            task_description=self.juggler_task_description, previous_status=prev_status
        )
        self.send_status_to_juggler(status="CRIT", description=description)
        super(NocdevCmdbSyncFromRacktables, self).on_failure(prev_status)

    def on_timeout(self, prev_status):
        description = "{task_description}: previous {previous_status} -> current TIMEOUT".format(
            task_description=self.juggler_task_description, previous_status=prev_status
        )
        self.send_status_to_juggler(status="CRIT", description=description)
        super(NocdevCmdbSyncFromRacktables, self).on_timeout(prev_status)

    def on_terminate(self):
        description = "{task_description}: TERMINATED".format(task_description=self.juggler_task_description)
        self.send_status_to_juggler(status="CRIT", description=description)
        super(NocdevCmdbSyncFromRacktables, self).on_terminate()

    @property
    def juggler_task_description(self):
        return "{sandbox_type} {sandbox_url} created at {create_date} is in {sandbox_status} status".format(
            sandbox_type=self.type,
            sandbox_url="https://sandbox.yandex-team.ru/task/{}".format(self.id),
            create_date=self.created.isoformat(),
            sandbox_status=self.status,
        )

    def send_status_to_juggler(self, status, description=None):
        if not self.Parameters.report_to_juggler:
            return

        if description is None:
            description = self.juggler_task_description

        sync_err_log = self.log_path("cmdb.err.log")
        if sync_err_log.exists():
            lines = sync_err_log.read_text().splitlines(False)
            if lines:
                description += "\nLast error: " + lines[-1]

        logger.info("Sending status to juggler: {}; {}".format(status, description))

        jclient.send_events_to_juggler(
            self.Parameters.juggler_host,
            self.Parameters.juggler_service,
            status,
            description if description is not None else self.juggler_task_description,
        )
