# coding=utf-8
import os.path
import itertools
import logging
import sys
from datetime import datetime

from sandbox import common
from sandbox import sdk2
from sandbox.common.types import task
from sandbox.common.types.misc import DnsType
from sandbox.projects.metrika import utils
from sandbox.projects.ofd.metrika_fork.utils.pipeline.common import DTF
from sandbox.projects.ofd.metrika_fork.utils.pipeline.pipeline_errors import PipelineInternalError, PipelineFailure, \
    PipelineError, PipelineAbortError
from sandbox.projects.ofd.metrika_fork.utils.pipeline.pipeline_state import StateManager
from sandbox.projects.ofd.metrika_fork.utils.pipeline.pipeline_view import StateViewModel
from sandbox.projects.ofd.metrika_fork.utils.pipeline.retry_memoize_creator import MemoizeCreator

logger = logging.getLogger('pipeline')


class Audit(object):
    # экземпляр конвейера создан - особой ценности не имеет
    CREATED = 'CREATED'

    # конвейер запущен - событие начала конвейера
    STARTED = 'STARTED'

    # промежуточное событие - начало стадии
    STAGE_STARTED = 'STAGE_STARTED'
    # промежуточное собыние - успешное завершение стадии
    STAGE_FINISHED = 'STAGE_FINISHED'

    # конвейер приостановлен из-за поломок в нём
    BROKEN = 'BROKEN'
    # конвейер приостановлен из-за поломок вне его
    EXCEPTION = 'EXCEPTION'

    # конвейер продолжен после одного из двух предыдущих событий
    RESUMED = 'RESUMED'

    # конвейер завершён успешно с позитивным результатом
    FAILURE = 'FAILURE'
    # конвейер завершён успешно с негативным результатом
    SUCCESS = 'SUCCESS'

    # конвейере прерван, результата достигнуто не будет
    ABORT = 'ABORT'


class PipelineBase(object):
    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64

    class Context(sdk2.Task.Context):
        state = {}

    def create_stages(self):
        """
        Должен быть перегружен в наследнике
        :return: iterable, каждый элемент которого представляет собой стадию, стадией может быть:
        имя функции или функция в классе задачи, либо iterable, первый элемент которого
        то, что указано выше, а второй - title стадии
        """
        return []

    def on_save(self):
        # тут строим пайплайн
        # в зависимости от значений параметров можно сделать по разному
        # по ходу валидируя, что есть все нужные функции у всех стадий, а стадии только что созданы и не завершены
        self.Context.state = StateManager().init(self.create_stages()).validate(self).state

    def on_enqueue(self):
        # тут делаем валидацию self.Context.state - есть ли все функции не завершённых стадий
        # почему тут нужно делать это - код задачи мог измениться.
        self.state.validate(self)

    def on_execute(self):
        # в бинарных тасках не запускаются on_save и on_enqueue
        if not self.Context.state:
            self.Context.state = StateManager().init(self.create_stages()).validate(self).state

        # проверяем, выполнялся ли конвейер
        # если да, то отправляем сообщение о запуске конвейера в БД
        # аудит - STARTED
        if not self.state.started:
            self.state.start()
            self.Context.save()
            self._audit(Audit.STARTED)

        # тут выполняем пайплайн
        # к этому моменту self.Context.state - провалидирован
        # получаем очередную стадию, узнаём, финальная ли она,
        # выполняем тем или иным образом, обновляем self.Context.state, идём далее (или выбрасываем исключение)
        while not self.state.is_completed:
            stage = self.state.next_stage
            # проверить до старта - сколько ретраев в стадии, если 0 - то начинаем новую - не шлём аудит
            # если более нуля и у последнего есть finish_ts, то это повтор после прерывания
            if stage.retry_count > 0 and stage.retries_wrapped[-1].finish_ts is not None:
                self._audit(Audit.RESUMED)
            stage.start(self.log_resource.id)
            self.Context.save()
            try:
                getattr(self, stage.func_name)()
            except (common.errors.Wait, common.errors.TemporaryError):
                logger.info("Sandbox flow control exception. Reraise", exc_info=sys.exc_info())
                raise
            except PipelineInternalError:
                stage.finish(False)
                self.Context.save()
                logger.error("Internal pipeline error. Please fix pipeline.", exc_info=sys.exc_info())
                # аудит - BROKEN
                self._audit(Audit.BROKEN)
                raise
            except PipelineError:
                stage.finish(False)
                self.Context.save()
                logger.warning("Retrieable pipeline error.", exc_info=sys.exc_info())
                # аудит - EXCEPTION
                self._audit(Audit.EXCEPTION)
                raise
            except PipelineFailure as e:
                if self.state.is_on_final_stage:
                    # пайплайн дошёл до конца с провалом, ретраев более не будет
                    stage.finish(True)
                    self.Context.save()
                    logger.info("Pipeline failed. Sorry.", exc_info=sys.exc_info())
                    raise
                else:
                    # это ошибка в логике пайплайна
                    stage.finish(False)
                    self.Context.save()
                    logger.warning("Pipeline failure not in final stage. Please fix pipeline.", exc_info=sys.exc_info())
                    # аудит - BROKEN
                    self._audit(Audit.BROKEN)
                    raise PipelineInternalError("Ошибка в логике пайплайна - это не финальная стадия", e)
            except PipelineAbortError:
                stage.finish(False)
                logger.info("Pipeline is not able to achieve any result.")
                raise
            except Exception as e:
                stage.finish(False)
                self.Context.save()
                logger.info("Other non special exception. Reraise as PipelineError.", exc_info=sys.exc_info())
                # аудит - EXCEPTION
                self._audit(Audit.EXCEPTION)
                self.set_info(PipelineError("Обнаружена общая ошибка.", e).get_task_info())
                raise

            # Стадия дошла до конца, успешно.
            stage.finish(True)
            self.Context.save()

    def on_finish(self, prev_status, status):
        # аудит - SUCCESS or FAILURE в зависимости от status
        if status == task.Status.SUCCESS:
            self._audit(Audit.SUCCESS)
        elif status == task.Status.FAILURE:
            if not self.state.is_completed:
                self._audit(Audit.ABORT)
            else:
                self._audit(Audit.FAILURE)
        else:
            logger.warning("Unsupported final status: {}".format(status))

        # отсылка статистики по стадиям
        self._stat(self.state)

    def on_before_timeout(self, seconds):
        self.Context.save()

    def timeout_checkpoints(self):
        return [0]

    @property
    def memoize_stage(self):
        """
        Изменяем поведение memoize_stage так, что бы оно работало в пределах одной повторной попытки
        :return:
        """
        return MemoizeCreator(self, self.state.current_stage.func_name, self.state.current_stage.retry_count)

    @property
    def memoize_stage_global(self):
        return super(PipelineBase, self).memoize_stage

    @property
    def state(self):
        return StateManager(self.Context.state)

    @property
    def view(self):
        return StateViewModel(self)

    def _audit(self, event):
        audit = {
            'EventTime': datetime.now().strftime(DTF),
            'author': self.author,
            'owner': self.owner,
            'task_id': self.id,
            'task_type': self.type.name,
            'event': event,
            'tags': [str(tag) for tag in list(self.Parameters.tags)],
            'params': ["{}={}".format(k, v) for k, v in dict(self.Parameters).iteritems()]
        }
        logger.info("DRY-RUN Audit: {}".format(audit))

    def _stat(self, state):
        stats = list(itertools.chain.from_iterable([[
            {
                'task_id': self.id,
                'StageName': stage.func_name,
                'StageTitle': stage.title,
                'StartTs': retry.start_ts,
                'FinishTs': retry.finish_ts,
                'RetryState': 'SUCCESS' if retry.is_completed else 'FAIL'
            } for retry in stage.retries_wrapped] for stage in state.stages_wrapped]))
        logger.info("DRY-RUN Stat: {}".format(stats))

    @sdk2.header()
    def header(self):
        if self.Context.state:
            return utils.render(
                "view.html.jinja2",
                {"state": self.view},
                dir=os.path.dirname(__file__),
            )
