# coding=utf-8

from datetime import datetime

from sandbox.common import config
from sandbox.common.types.task import Status
from sandbox.common.utils import execution_dir_link
from sandbox.projects.metrika import utils
from sandbox.projects.ofd.metrika_fork.utils.pipeline.common import DTF
from sandbox.projects.ofd.metrika_fork.utils.pipeline.pipeline_state import RetryManager, StageManager, StateManager
from sandbox.sdk2.service_resources import TaskLogs

SPAN_TEMPLATE = "<span class=\"status status_{}\">{}</span>"
GBR_SPAN_TEMPLATE = SPAN_TEMPLATE + "<a href=\"https://t.me/joinchat/Bvt2F0Ji2a_iwKDkMlrlFA\">Обратитесь в ГБР</a>"

NOT_STARTED = (Status.DRAFT,)
IN_PROGRESS = (Status.ENQUEUING, Status.ENQUEUED, Status.ASSIGNED, Status.PREPARING, Status.EXECUTING,
               Status.TEMPORARY, Status.FINISHING, Status.STOPPING, Status.SUSPENDING, Status.WAIT_MUTEX,
               Status.WAIT_OUT, Status.WAIT_RES, Status.WAIT_TASK, Status.WAIT_TIME, Status.RELEASING)
INTERVENTION_REQUIRED = (Status.EXCEPTION, Status.TIMEOUT, Status.STOPPED, Status.NO_RES, Status.EXPIRED)
FINISHED = (Status.SUCCESS, Status.FAILURE, Status.RELEASED)
ABORTED = (Status.FAILURE,)
BROKEN = (Status.SUCCESS, Status.RELEASING, Status.NOT_RELEASED, Status.RELEASED, Status.FAILURE, Status.DELETED)


class RetryViewModel(object):

    def __init__(self, task, retry_manager):
        self.task = task
        self.retry_manager = retry_manager

    @property
    def start_ts(self):
        return self.retry_manager.start_ts if self.retry_manager.start_ts else utils.decode("не начато")

    @property
    def finish_ts(self):
        return self.retry_manager.finish_ts if self.retry_manager.finish_ts else utils.decode("не завершено")

    @property
    def duration(self):
        try:
            return str(datetime.strptime(self.retry_manager.finish_ts, DTF) -
                       datetime.strptime(self.retry_manager.start_ts, DTF))
        except:
            return utils.decode("не определено")

    @property
    def current_status(self):
        """
        Текущее состояние повтора
        1. Не начат - такого не бывает.
        2. Начат
        3. Завершён - успешно
        4. Завершён - провален
        5. Приостановлен - когда таска в STOPPED
        6. Сломан - во всех прочих случаях
        :return: строка html для вставки в шаблон
        """
        if self.retry_manager.finish_ts:
            if self.retry_manager.is_completed:
                # завершён пройдено SUCCESS
                return utils.decode(SPAN_TEMPLATE.format("success", "успешно"))
            else:
                # завершён не пройдено FAILURE
                return utils.decode(SPAN_TEMPLATE.format("failure", "провалено"))
        else:
            if self.task.status in IN_PROGRESS:
                # начат EXECUTING
                return utils.decode(SPAN_TEMPLATE.format("executing", "в процессе"))
            elif self.task.status in INTERVENTION_REQUIRED:
                return utils.decode(SPAN_TEMPLATE.format("failure", "приостановлено"))
            else:
                return utils.decode(SPAN_TEMPLATE.format("failure", "сломано"))

    @property
    def logs(self):
        """
        Возвращает список html-ссылок на логи, связанные с данной попыткой
        :return:
        """

        def get_proxy():
            return config.Registry().client.fileserver.proxy.host

        def get_url(task_id, log_id):
            if get_proxy():
                return "https://{}/{}".format(get_proxy(), log_id)
            else:
                return "{}/{}".format(execution_dir_link(task_id), get_path_name(log_id))

        def get_path_name(id):
            r = TaskLogs.find(id=id).first()
            return r.path.name if r else id

        return ["<a href=\"{}\">{}</a>".format(get_url(self.task, log_id), get_path_name(log_id))
                for log_id in self.retry_manager.logs]


class StageViewModel(object):
    def __init__(self, task, stage_manager):
        self.task = task
        self.stage_manager = stage_manager

    @property
    def title(self):
        return self.stage_manager.title

    @property
    def current_status(self):
        """
        Текущее состояние стадии
        1. Не начата
        2. В процессе
        3. В процессе, требует вмешательства (EXCEPTION, TIMEOUT, STOPPED, NO_RES, EXPIRED) -  группа BREAK
        4. Пройдена
        :return: строка html для вставки в шаблон
        """
        if self.stage_manager.is_completed:
            # пройдена
            return utils.decode(SPAN_TEMPLATE.format("success", "завершено"))
        else:
            if self.stage_manager.start_ts:
                if self.task.status in IN_PROGRESS:
                    # в процессе
                    return utils.decode(SPAN_TEMPLATE.format("executing", "в процессе"))
                elif self.task.status in INTERVENTION_REQUIRED:
                    # в процессе, требует вмешательства
                    return utils.decode(SPAN_TEMPLATE.format("failure", "требует вмешательства", self.duration))
                elif self.task.status in ABORTED:
                    # прервано
                    return utils.decode(SPAN_TEMPLATE.format("failure", "прервано", self.duration))
                else:
                    return utils.decode(GBR_SPAN_TEMPLATE.format("failure", "Конвейер сломан немыслимым образом"))
            else:
                # не начата
                return utils.decode("")

    @property
    def start_ts(self):
        return self.stage_manager.start_ts if self.stage_manager.start_ts else utils.decode("не начато")

    @property
    def finish_ts(self):
        return self.stage_manager.finish_ts if self.stage_manager.finish_ts else utils.decode("не завершено")

    @property
    def duration(self):
        try:
            return str(datetime.strptime(self.stage_manager.finish_ts, DTF) -
                       datetime.strptime(self.stage_manager.start_ts, DTF))
        except:
            return utils.decode("")

    @property
    def retries(self):
        return [RetryViewModel(self.task, RetryManager(retry_state)) for retry_state in self.stage_manager.retries]

    @property
    def last_retry(self):
        return RetryViewModel(self.task, RetryManager(
            self.stage_manager.retries[-1])) if self.stage_manager.retry_count > 0 else None


class StateViewModel(object):
    """
    Представление для рендеринга custom header'а
    """

    def __init__(self, task):
        self.task = task
        self.state_manager = StateManager(task.Context.state)

    @property
    def current_status(self):
        """
        текущее состояние пайплайна.
        1. Не запущено NOT_STARTED
        2. В процессе IN_PROGRESS
        3. В процессе, требует вмешательства INTERVENTION_REQUIRED
        4. Прервано - BROKEN, но финальная стадия не завершена - это не должно наблюдаться, значит всё сломано
                   обратитесь в ГБР
        5. Завешено - FINISHED и финальная стадия завершена, результат пайплайна получен/определён
            1. Позитивный результат - SUCCESS -, - RELEASED -
            2. Негативный результат - FAILURE -
        6. Прервано - ABORTED и финальная стадия не завершена, результат пайплайна не может быть достигнут, но в
                ГБР обращаться не нужно
        :return: строка html для вставки в шаблон
        """
        if self.task.status in NOT_STARTED:
            # не запущен
            return utils.decode(SPAN_TEMPLATE.format("draft", "Конвейер не запущен"))
        elif self.task.status in IN_PROGRESS:
            return utils.decode(SPAN_TEMPLATE.format("executing", "Конвейер запущен"))
        elif self.task.status in INTERVENTION_REQUIRED:
            return utils.decode(SPAN_TEMPLATE.format("failure", "Конвейер требует вмешательства"))
        elif self.task.status in FINISHED and self.state_manager.is_completed:
            if self.task.status in {Status.SUCCESS, Status.RELEASED}:
                return utils.decode(SPAN_TEMPLATE.format("success", "Конвейер завершён с позитивным результатом"))
            elif self.task.status == Status.FAILURE:
                return utils.decode(SPAN_TEMPLATE.format("failure", "Конвейер завершён с негативным результатом"))
        elif self.task.status in ABORTED and not self.state_manager.is_completed:
            return utils.decode(SPAN_TEMPLATE.format("failure", "Конвейер прерван, результат не может быть получен"))
        elif self.task.status in BROKEN and not self.state_manager.is_completed:
            return utils.decode(GBR_SPAN_TEMPLATE.format("failure", "Конвейер сломан"))
        else:
            return utils.decode(GBR_SPAN_TEMPLATE.format("failure", "Конвейер сломан немыслимым образом"))

    @property
    def result(self):
        """
        Итоговый результат работы пайплайна. Пуст, пока не выполнился последний этап.
        :return: строка html для вставки в шаблон
        """

        if self.state_manager.is_completed:
            if self.task.status in IN_PROGRESS:
                return utils.decode("")
            elif self.task.status in {Status.SUCCESS, Status.RELEASED, Status.RELEASING}:
                return utils.decode(SPAN_TEMPLATE.format("success", "Успех"))
            else:
                return utils.decode(SPAN_TEMPLATE.format("failure", "Провал"))
        else:
            return utils.decode("")

    @property
    def start_ts(self):
        """
        Время начала пайплайна
        :return: строка html для вставки в шаблон
        """
        return self.state_manager.start_ts if self.state_manager.start_ts else utils.decode("")

    @property
    def finish_ts(self):
        """
        Время завершения пайплайна
        :return: строка html для вставки в шаблон
        """
        return self.state_manager.finish_ts if self.state_manager.finish_ts else utils.decode("")

    @property
    def duration(self):
        """
        Текущая длительность
        :return:
        """
        try:
            return str(datetime.strptime(self.state_manager.finish_ts, DTF) -
                       datetime.strptime(self.state_manager.start_ts, DTF))
        except:
            return utils.decode("-")

    @property
    def stages(self):
        """
        iterable c viewmodel'ями стадий конвейера
        :return:
        """
        return [StageViewModel(self.task, StageManager(stage_state)) for stage_state in self.state_manager.stages]
