# coding=utf-8
from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.ofd.metrika_fork.utils.pipeline.pipeline import PipelineBase
from sandbox.projects.ofd.ofd_release.release_helper import ReleaseHelper
from sandbox.projects.ofd.ofd_release.release_info import ReleaseInfo
from sandbox.projects.ofd.ofd_release.settings import Settings
from sandbox.sdk2 import Parameters, Requirements
from sandbox.sdk2.parameters import Bool, List, Staff, String
from sandbox.sandboxsdk.environments import PipEnvironment


class OfdRelease(PipelineBase, sdk2.Task):
    """
    Релиз ОФД
    """

    @property
    def release(self):
        return ReleaseInfo(self.Context.release_info_state)

    @property
    def settings(self):
        return Settings(owner=self.Parameters.secret_owner, login=self.Parameters.operator)

    class Context(PipelineBase.Context):
        release_info_state = ReleaseInfo().state

    class Requirements(Requirements):
        dns = DnsType.DNS64
        environments = (
            PipEnvironment("requests", version="2.18.4", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("yandex-tracker-client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"]),
        )

    class Parameters(Parameters):
        owner = "METRIKA"

        description = "Релиз ОФД"

        service = String("Сервис", required=True, default=ReleaseInfo().service_release_info_dict.keys()[1], choices=[(key, key) for key in ReleaseInfo().service_release_info_dict.keys()],
                         description="Сервис, релиз которого будет осуществлен.")

        revision_id = String("Ревизия в Аркадии", required=True, default="", description="Номер ревизии в Аркадии, из которой будут собираться пакеты.")

        apply_patch = Bool("Применить патч", default=False)
        with apply_patch.value[True]:
            review_id = List("Ревью в Аркануме", default=[], description="ID ревью в Аркануме, патч из которого будет применяться в сборке пакетов.")

        create_issue = Bool("Создать релизный тикет", default=False)
        with create_issue.value[False]:
            issue_key = String("Релизный тикет", required=True, default="", description="Номер релизного тикета в очереди OFD, например OFD-9999")

        is_new = Bool("Новый релиз", required=True, default=True, description="Начать ли новый релиз. У старого релиза будет увеличиваться патч.")

        with is_new.value[False]:
            previous_release_version = String("Версия предыдущего релиза", required=True, default="", description="Полная версия предыдущего релиза указанного сервиса.")

        secret_owner = String("Owner",
                              default="METRIKA",
                              description="Владелец секретов, используемых при операциях")

        with String("Owner", default="OFD", required=True) as secret_owner:
            for owner in ('METRIKA', 'MAIL', 'OFD'):
                secret_owner.values[owner] = owner

        operator = Staff("Operator",
                          default='robot-ofd',
                          required=True,
                          description="Логин корпового сотрудника (предпочтительно робота), от имени которого будут осуществляться операции деплоя")

    def on_enqueue(self):
        self.release.author = self.author
        self.release.select_service(self.Parameters.service)
        self.release.previous_release_version = self.Parameters.previous_release_version
        self.release.release_version = ReleaseHelper.build_version(1, self.id, 1) if self.Parameters.is_new else ReleaseHelper.increase_version(self.release.previous_release_version, False)

        ReleaseHelper.update_description(self, self.release.author, [self.Parameters.service], self.release.release_version)

    def create_stages(self):
        return [
            (self.stage_create_release_issue, "Релизный тикет"),
            (self.stage_build, "Сборка"),
            (self.stage_deployment_to_testing, "Установка в тестинг")
        ]

    def stage_create_release_issue(self):
        if self.Parameters.create_issue:
            ReleaseHelper.create_release_issue(self, self.release, self.settings)
        else:
            self.release.release_issue_key = self.Parameters.issue_key

        ReleaseHelper.set_create_release_issue_info(self, self.release)

    def stage_build(self):
        with self.memoize_stage.stage_build(commit_on_entrance=False):
            ReleaseHelper.set_build_info(self, self.release)
            self.Context.build_task_id = ReleaseHelper.build(self, self.release)
            ReleaseHelper.wait_tasks(self.Context.build_task_id)

        ReleaseHelper.check_task(self, self.Context.build_task_id)

    def stage_deployment_to_testing(self):
        ReleaseHelper.deploy_to_testing(self, self.release, self.settings)

        ReleaseHelper.set_deploy_to_testing_info(self, self.release)
