# coding=utf-8
import logging
from distutils.version import StrictVersion

from sandbox.projects.metrika.java.metrika_java_release import release_helper
from sandbox.projects.metrika.utils import settings as metrika_settings
from sandbox.projects.ofd.ofd_release.conductor import Conductor
from sandbox.sdk2 import Task
from sandbox.sdk2 import Vault

logger = logging.getLogger("release-helper")


class ReleaseHelper(release_helper.ReleaseHelper):

    @staticmethod
    def get_release_main_version(full_version):
        major, minor, patch = StrictVersion(full_version).version
        return "{}.{}".format(major, minor)

    @staticmethod
    def create_release_issue(task, release, settings=metrika_settings):
        from startrek_client import Startrek
        st_client = Startrek(token=Vault.data(settings.owner, settings.tracker_token), useragent=settings.owner)

        search_parameters = {
            "queue": release.service.release_issue_queue,
            "components": release.service.release_issue_component,
            "tags": ReleaseHelper.get_release_main_version(release.release_version),
            "type": "release"
        }
        create_parameters = search_parameters.copy()
        create_parameters.update({
            "summary": "Релиз {} {}".format(release.service.name, ReleaseHelper.get_release_main_version(release.release_version)),
            "assignee": release.author,
            "followers": release.service.release_issue_followers
        })

        issue = ReleaseHelper.find_or_create_issue(st_client, search_parameters, create_parameters, task.id)
        logger.info("Issue {} created.".format(issue.key))
        release.release_issue_key = issue.key

    @staticmethod
    def build(task, release):
        return Task[release.service.build_task](
            task,
            description="Release {} build: {}".format(release.release_version, release.service.name),
            arcadia_path="ofd/{}".format(release.service.name),
            package_version=release.release_version,
            apply_revision=True,
            revision_id=task.Parameters.revision_id,
            apply_patch=task.Parameters.apply_patch,
            review_id=task.Parameters.review_id,
            publisher=task.Parameters.operator,
            run_tests=False,
            upload_to_dist=True
        ).enqueue().id

    @staticmethod
    def deploy_to_testing(task, release, settings):
        conductor = Conductor(task, settings)
        packages = {release.service.build_name: release.release_version}
        release.conductor_ticket = conductor.create_conductor_ticket(packages, release.release_issue_key, branch=release.service.conductor_init_branch)
        conductor.wait_conductor_tickets([release.conductor_ticket])

    @staticmethod
    def set_create_release_issue_info(task, release):
        create_release_issue_info = "Задача:<br/><a href=\"https://st.yandex-team.ru/{0}\">{0}</a>".format(release.release_issue_key)

        task.set_info(create_release_issue_info, do_escape=False)

    @staticmethod
    def set_build_info(task, release):
        build_info = "Пакет:<br/><a href=\"https://dist.yandex.ru/find?pkg={0}&repo=ofd&ver={1}&human=True\">{0}={1}</a>".format(release.service.build_name, release.release_version)

        task.set_info(build_info, do_escape=False)

    @staticmethod
    def set_deploy_to_testing_info(task, release):
        deploy_to_testing_info = "Кондукторный тикет (<a href=\"https://rackroll.mtrs.yandex-team.ru/rackroll/conductor/tasks\">Rack & Roll</a>):<br/>" \
                                 "<a href=\"https://c.yandex-team.ru/tickets/{0}\">{0}</a>".format(release.conductor_ticket)

        task.set_info(deploy_to_testing_info, do_escape=False)
