# coding=utf-8

import os
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import DnsType
from sandbox.projects.resource_types import TASK_CUSTOM_LOGS
from sandbox.projects.metrika.utils.mixins.console import BaseConsoleMixin
from sandbox.projects.metrika.utils.artifacts import list_artifacts, archive_artifacts
from sandbox.projects.metrika.admins.metrika_packages_upload import MetrikaPackagesUpload
from sandbox.projects.ofd.runtime.ofd_runtime_run_tests import OfdRuntimeRunTests
from sandbox.projects.ofd.common import VcsParameters


class OfdRuntimePackageBuild(BaseConsoleMixin, sdk2.Task):
    """
    Собирает ofd-runtime пакет
    """

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64
        disk_space = 1024

    class Context(sdk2.Task.Context):
        artifact_resource_id = None
        upload_task_id = None
        tests_task_id = None

    class Parameters(sdk2.Task.Parameters):
        vcs_block = VcsParameters()
        container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=806900009,
            required=True,
        )
        with sdk2.parameters.Group("build") as tests_block:
            package_version = sdk2.parameters.String("Package version", default="", description="Версия пакета")
            apply_revision = sdk2.parameters.Bool("Собрать из ревизии", default=False)
            with apply_revision.value[True]:
                revision_id = sdk2.parameters.String("Revision id", default="", description="Номер ревизии в аркадии")
            apply_patch = sdk2.parameters.Bool("Применить патч", default=False)
            with apply_patch.value[True]:
                review_id = sdk2.parameters.List("Review id", default=[], description="ID ревью в аркануме")
            run_tests = sdk2.parameters.Bool("Запускать тесты", default=True)
            upload_to_dist = sdk2.parameters.Bool("Загружать на dist", default=False)
            with upload_to_dist.value[True]:
                repository = sdk2.parameters.String("Репозиторий",
                                                    required=True,
                                                    default="ofd")

    def on_execute(self):
        if self.Parameters.run_tests:
            with self.memoize_stage.run_tests(commit_on_entrance=False):
                self._run_tests()
            with self.memoize_stage.check_tests(commit_on_entrance=False):
                self._check_tests()

        with self.memoize_stage.build(commit_on_entrance=False):
            with sdk2.helpers.ProgressMeter("Checkout"):
                self._checkout_branch()
            if self.Parameters.apply_patch:
                with sdk2.helpers.ProgressMeter("Patch"):
                    for rid in self.Parameters.review_id:
                        sdk2.vcs.svn.Arcadia.fetch_patch('rb:{}'.format(rid), self._work_dir())
                        sdk2.vcs.svn.Arcadia.apply_patch_file(self._work_dir(), os.path.join(self._work_dir(), 'arcadia.patch'), is_zipatch=True)
            try:
                with sdk2.helpers.ProgressMeter("Build"):
                    self._build()
            finally:
                with sdk2.helpers.ProgressMeter("Archive Artifacts"):
                    archive_artifacts(self, self._work_dir(), "build_logs", TASK_CUSTOM_LOGS,
                                            *list_artifacts(self._work_dir(), '\.log$'))

            with sdk2.helpers.ProgressMeter("Archive Artifacts"):
                self.Context.artifact_resource_id = archive_artifacts(
                    self, os.path.join(self._work_dir(), "packages"), "artifacts", TASK_CUSTOM_LOGS,
                    *list_artifacts(os.path.join(self._work_dir(), "packages"), '\.(deb|changes)$')
                )

        if self.Parameters.upload_to_dist:
            with self.memoize_stage.upload(commit_on_entrance=False):
                self._upload_to_dist()
            with self.memoize_stage.check_upload(commit_on_entrance=False):
                self._check_upload()

    def _work_dir(self, *path):
        return str(self.path("wd", *path))

    def _checkout_branch(self):
        project_url = os.path.join(self.Parameters.arcadia_url, self.Parameters.arcadia_path)
        if self.Parameters.apply_revision:
            sdk2.vcs.svn.Arcadia.checkout(project_url, self._work_dir(), revision=self.Parameters.revision_id)
        else:
            sdk2.vcs.svn.Arcadia.checkout(project_url, self._work_dir())

    def _build(self):
        script = '''#!/bin/bash -xe
'''
        if self.Parameters.package_version:
            script += 'dch -v "{}" -D "trusty" "all latest updates" > debuild.log 2>&1'.format(self.Parameters.package_version)
        self._execute_script(script + '''
dpkg-buildpackage -b -d -uc >> debuild.log 2>&1
mkdir packages
mv ../*.deb packages/
mv ../*.changes packages/
''', cwd=self._work_dir())

    def _run_tests(self):
        tests_task = OfdRuntimeRunTests(self)
        tests_task.Parameters.arcadia_url = self.Parameters.vcs_block.arcadia_url
        tests_task.Parameters.arcadia_path = self.Parameters.vcs_block.arcadia_path
        tests_task.Parameters.vcs = self.Parameters.vcs_block.vcs
        tests_task.Parameters.container = self.Parameters.container
        tests_task.Parameters.apply_patch = self.Parameters.apply_patch
        tests_task.Parameters.apply_revision = self.Parameters.apply_revision
        if self.Parameters.apply_patch:
            tests_task.Parameters.review_id = self.Parameters.review_id
        if self.Parameters.apply_revision:
            tests_task.Parameters.revision_id = self.Parameters.revision_id
        tests_task.save()

        self.Context.tests_task_id = tests_task.id
        tests_task.enqueue()
        raise sdk2.WaitTask(self.Context.tests_task_id, tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK))

    def _upload_to_dist(self):
        upload_task = MetrikaPackagesUpload(self, **{
            MetrikaPackagesUpload.Parameters.artifacts.name: self.Context.artifact_resource_id,
            MetrikaPackagesUpload.Parameters.repository.name: self.Parameters.repository
        })
        self.Context.upload_task_id = upload_task.id
        upload_task.enqueue()
        raise sdk2.WaitTask(self.Context.upload_task_id, tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK))

    def _check_upload(self):
        if self.Context.upload_task_id:
            status = self.find(id=self.Context.upload_task_id).first().status
            if status not in ctt.Status.Group.SUCCEED:
                raise TaskFailure("Child task status: {}".format(status))

    def _check_tests(self):
        if self.Context.tests_task_id:
            status = self.find(id=self.Context.tests_task_id).first().status
            if status not in ctt.Status.Group.SUCCEED:
                raise TaskFailure("Child task status: {}".format(status))
