# coding=utf-8

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.types.misc import DnsType
from sandbox.projects.common.build import YaPackage
from sandbox.projects.common.build import parameters
from sandbox.projects.metrika.utils.mixins.console import BaseConsoleMixin
from sandbox.projects.ofd.common import VcsParameters


class OfdTasksPackageArcadiaBuild(BaseConsoleMixin, sdk2.Task):
    """
    Собирает ofd-tasks пакет с аркадийной сборкой
    """

    @property
    def ya_package2(self):
        return sdk2.Task[YaPackage.YaPackage.type]

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64
        disk_space = 1024

    class Context(sdk2.Task.Context):
        child_tasks_ids = []

    class Parameters(sdk2.Task.Parameters):
        vcs_block = VcsParameters()
        with sdk2.parameters.Group("build") as tests_block:
            package_version = sdk2.parameters.String("Package version", default="", description="Версия пакета")
            apply_revision = sdk2.parameters.Bool("Собрать из ревизии", default=False)
            with apply_revision.value[True]:
                revision_id = sdk2.parameters.String("Revision id", default="", description="Номер ревизии в аркадии")
            apply_patch = sdk2.parameters.Bool("Применить патч", default=False)
            with apply_patch.value[True]:
                review_id = sdk2.parameters.List("Review id", default=[], description="ID ревью в аркануме")
            run_tests = sdk2.parameters.Bool("Запускать тесты", default=False)
            upload_to_dist = sdk2.parameters.Bool("Загружать на dist", default=False)
            with upload_to_dist.value[True]:
                repository = sdk2.parameters.String("Репозиторий",
                                                    required=True,
                                                    default="ofd")
            publisher = sdk2.parameters.String("Publisher", default='robot-ofd')

    def on_execute(self):
        with self.memoize_stage.generate_subtasks(commit_on_entrance=False):
            self.Context.child_tasks_ids.append(self._create_subtask_ya_package())

            raise sdk2.WaitTask(
                self.Context.child_tasks_ids,
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
            )

        with self.memoize_stage.get_child_statuses(commit_on_entrance=True):
            failed_subtasks = []
            for subtask in self.find():
                if subtask.status != ctt.Status.SUCCESS:
                    failed_subtasks.append(subtask)
            if failed_subtasks:
                raise TaskError("Failed subtasks: %s" % [s.id for s in failed_subtasks])

    def _create_subtask_ya_package(self):
        ya_pkg_ctx = {
            YaPackage.PackagesParameter.name: 'ofd/tasks/package.json',
            YaPackage.PackageTypeParameter.name: YaPackage.DEBIAN,
            YaPackage.ChangelogParameter.name: self.Parameters.description,
            YaPackage.KeyUserParameter.name: self.Parameters.publisher,
            YaPackage.PublishToParameter.name: self.Parameters.repository,
            YaPackage.PublishPackageParameter.name: self.Parameters.upload_to_dist,
            YaPackage.RunTestsParameter.name: self.Parameters.run_tests,
            YaPackage.CustomVersionParameter.name: self.Parameters.package_version,
            parameters.UseArcadiaApiFuse.name: False,
            parameters.ArcadiaUrl.name: self.Parameters.arcadia_url,
            'description': 'ofd-tasks-arcadia build, child of %d' % self.id,
        }
        if self.Parameters.apply_patch:
            ya_pkg_ctx[parameters.ArcadiaPatch.name] = "arc:%s" % self.Parameters.review_id[0]

        child = self.ya_package2(
            self,
            **ya_pkg_ctx
        )  # type: sdk2.Task

        child.Requirements.disk_space = 1024
        child.save()
        child.enqueue()

        return child.id
