import logging
import os
import uuid
from multiprocessing import cpu_count
from multiprocessing.dummy import Pool
from operator import attrgetter

from sandbox import sdk2
from sandbox.projects.offerwall import OfferwallImages
from sandbox.projects.offerwall.common.tasks import SQLAlchemyTask
from sandbox.projects.offerwall.common.utils import (
    Enum, LogS3Retrieve, RetryRetrieve, S3Retrieve, cached_property
)

logger = logging.getLogger('offerwall.images')

OfferStatus = Enum('OfferStatus', 'ACTIVE INACTIVE REMOVED DRAFT', start=1)
IMAGES_PATH = 'images'


class OfferwallImagesTask(SQLAlchemyTask):
    class Requirements(SQLAlchemyTask.Requirements):
        disk_space = 1024

    @cached_property
    def declared_models(self):
        from sqlalchemy import Column, Integer, String, Boolean
        from sqlalchemy.dialects.postgresql import UUID
        from sqlalchemy.ext.declarative import declarative_base
        from sqlalchemy.orm import relationship

        Base = declarative_base()

        class Offer(Base):
            __tablename__ = 'core_offer'

            id = Column(UUID, primary_key=True, default=uuid.uuid4)
            status = Column(Integer)
            is_active = Column(Boolean)

        class Info(Base):
            __tablename__ = 'core_info'

            id = Column(Integer, primary_key=True)

            offer = relationship(Offer)

        class Images(Base):
            __tablename__ = 'core_images'

            id = Column(Integer, primary_key=True)

            banner = Column(String)
            info = relationship(Info)

        return Offer, Info, Images

    @cached_property
    def retrieve(self):
        if not os.path.exists(IMAGES_PATH):
            os.mkdir(IMAGES_PATH)

        return LogS3Retrieve(
            retrieve=RetryRetrieve(
                retrieve=S3Retrieve(
                    endpoint_url=self.Parameters.aws_endpoint_url,
                    bucket=self.Parameters.aws_bucket,
                    path=IMAGES_PATH
                ),
                retry=3,
                exc_types=(IOError,)
            ),
            logger=logger
        )

    def on_execute(self):
        from sqlalchemy import select

        Offer, _, Images = self.declared_models
        query = select([Images.banner]).where(
            Offer.is_active.is_(True)).where(
            Offer.status.in_([OfferStatus.ACTIVE, OfferStatus.INACTIVE])
        )
        images = self.connection.execute(query).fetchall()
        banners = map(attrgetter('banner'), images)
        Pool(processes=cpu_count() * 2).map(self.retrieve, banners)

        if os.listdir(IMAGES_PATH):
            sdk2.ResourceData(
                OfferwallImages(
                    task=self,
                    description='',
                    path=IMAGES_PATH,
                    released=self.Parameters.release_type,
                    ttl=self.Parameters.ttl
                )
            ).ready()
