import urllib

from sandbox import sdk2
from sandbox.sandboxsdk.sandboxapi import RELEASE_STABLE, RELEASE_TESTING
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.offerwall.common.environments import (
    YandexInternalRootCAEnvironment)
from sandbox.projects.offerwall.common.utils import (
    cached_property, parse_credentials)


class SQLAlchemyTask(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        environments = (
            PipEnvironment('sqlalchemy'),
            PipEnvironment('psycopg2-binary'),
            YandexInternalRootCAEnvironment()
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 10 * 60

        vault_db_credentials_name = sdk2.parameters.String(
            label='Vault DB credentials name', required=True)

        with sdk2.parameters.Group('AWS settings') as aws_block:
            aws_endpoint_url = sdk2.parameters.String(
                label='Endpoint url', required=True)
            aws_bucket = sdk2.parameters.String(label='Bucket', required=True)

        with sdk2.parameters.RadioGroup('Release', required=True) as release_type:
            release_type.values[RELEASE_TESTING] = release_type.Value(
                RELEASE_TESTING, default=True)
            release_type.values[RELEASE_STABLE] = release_type.Value(
                RELEASE_STABLE)

        ttl = sdk2.parameters.Integer(label='TTL', default=7)

    @cached_property
    def db_conn_url(self):
        db_conn_template = 'postgresql+psycopg2://{user}:{password}@{host}:{port}/{name}'
        settings = {
            'sslmode': 'verify-full',
            'sslrootcert': 'root.crt',
            'target_session_attrs': 'read-write'
        }
        db_conn_template += '?' + urllib.urlencode(settings)

        vault_data = sdk2.Vault.data(self.Parameters.vault_db_credentials_name)
        db_conn_url = db_conn_template.format(**parse_credentials(vault_data))

        return db_conn_url

    @cached_property
    def engine(self):
        from sqlalchemy import create_engine

        engine = create_engine(self.db_conn_url)
        return engine

    @cached_property
    def connection(self):
        return self.engine.connect()

    def make_session(self):
        from sqlalchemy.orm import sessionmaker

        return sessionmaker(self.engine)()
