from abc import ABCMeta, abstractmethod
from collections import namedtuple
from csv import DictReader, Sniffer
from urllib import urlretrieve
from urlparse import urljoin
import os


# noinspection PyPep8Naming
class cached_property(object):

    def __init__(self, func):
        self.__wrapped__ = func

    def __get__(self, instance, cls):
        if instance is None:
            return self

        result = self.__wrapped__(instance)
        instance.__dict__[self.__wrapped__.__name__] = result
        return result


# noinspection PyPep8Naming
class StickyProperty(object):
    def __init__(self, value):
        self.__value = value
        self.__changed = False

    def __get__(self, instance, cls):
        if instance is None:
            return self
        return self.__value

    def __set__(self, instance, value):
        if self.__value == value or self.__changed:
            return
        self.__value = value
        self.__changed = True


def class2dict(cls):
    return dict(
        (key, getattr(cls, key)) for key in dir(cls)
        if not key.startswith('__')
    )


def parse_credentials(text):
    return dict(item.split('=') for item in text.split())


# noinspection PyPep8Naming
def Enum(name, field_names, text=False, start=0):
    if text:
        return namedtuple(name, field_names)(*field_names.split())
    return namedtuple(name, field_names)(
        *range(start, len(field_names.split()) + start)
    )


def guess_dialect(csv_file):
    dialect = Sniffer().sniff(csv_file.read(1024))
    csv_file.seek(0)
    return dialect


class Reader(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def has_next(self):
        pass

    @abstractmethod
    def read(self, size):
        pass

    @abstractmethod
    def read_item(self):
        pass


class CSVReader(Reader):
    _has_next = StickyProperty(True)

    def __init__(self, file_, schema=None, size=None):
        if isinstance(file_, (str, unicode)):
            file_ = open(file_)
        self._reader = DictReader(file_, dialect=guess_dialect(file_))
        self._schema = schema
        self._size = size
        if hasattr(self._schema, 'load') and callable(self._schema.load):
            self._load = self._schema.load
        else:
            # noinspection PyPackageRequirements
            from marshmallow.schema import UnmarshalResult
            self._load = lambda x: UnmarshalResult(data=x, errors={})

    def has_next(self):
        return self._has_next

    def read_item(self):
        return self._load(self._reader.next())

    def read(self, size=None):
        size = int(size or self._size)

        if not self.has_next():
            raise StopIteration

        bucket = []
        try:
            for _ in xrange(size):
                item = self.read_item()
                if not item.errors:
                    bucket.append(item.data)
        except StopIteration:
            self._has_next = False
        return bucket


class LogErrorReader(Reader):
    def __init__(self, reader, logger):
        self._reader = reader
        self._logger = logger

    def has_next(self):
        return self._reader.has_next()

    def read_item(self):
        item = self._reader.read_item()
        if item.errors:
            self._logger.info(item.errors)
        return item

    def read(self, size=None):
        return self._reader.read(size)


class Retrieve(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def url(self, name):
        pass

    @abstractmethod
    def __call__(self, name):
        pass


class S3Retrieve(Retrieve):
    def __init__(self, endpoint_url, bucket, path):
        self._endpoint_url = endpoint_url
        self._bucket = bucket
        if not self._bucket.endswith('/'):
            self._bucket += '/'
        self._path = path

    def url(self, name):
        return urljoin(urljoin(self._endpoint_url, self._bucket), name)

    def __call__(self, name):
        path = os.path.join(self._path, name)
        return urlretrieve(self.url(name), path)[0]


class LogS3Retrieve(Retrieve):
    def __init__(self, retrieve, logger):
        self._retrieve = retrieve
        self._logger = logger

    def url(self, name):
        self._logger.info('Image name: {}'.format(name))
        url = self._retrieve.url(name)
        self._logger.info('Url is: {}'.format(url))
        return url

    def __call__(self, *args, **kwargs):
        return self._retrieve(*args, **kwargs)


class RetryRetrieve(Retrieve):
    def __init__(self, retrieve, retry, exc_types):
        self._retrieve = retrieve
        self._retry = retry
        self._exc_types = exc_types

    def url(self, name):
        return self._retrieve.url(name)

    def __call__(self, *args, **kwargs):
        last_exc = None
        for _ in range(self._retry):
            try:
                return self._retrieve(*args, **kwargs)
            except self._exc_types as e:
                last_exc = e

        raise last_exc
