# -*- coding: utf-8 -*-

import logging
import os

import requests
from requests.models import HTTPError

from sandbox.projects.common.arcadia.sdk import mount_arc_path

logger = logging.getLogger(__name__)


class ArcadiaDownloader(object):
    @staticmethod
    def get_files():
        logger.debug("Getting files from Arcadia")
        records = ArcadiaDownloader._find_package_jsons()
        result = []

        with mount_arc_path("arcadia-arc:/#trunk") as arcadia_path:
            logger.debug("Mounted Arc")
            for row in records:
                record = ArcadiaDownloader._get_file(arcadia_path, row)

                if record is not None:
                    result.append(record)

        logger.info("Got %s files from Arc", len(result))
        return result

    @staticmethod
    def _find_package_jsons():
        r = requests.get('http://codesearch.yandex-team.ru/search?file_regex=^.*/package.json$&max=999999999&branch=arcadia&format=json')

        try:
            r.raise_for_status()
        except HTTPError as e:
            logger.debug('Codesearch request failed!\n{}'.format(e))
            raise e

        data = r.json()
        return [{
            "commit": None,
            "file": x['FileChunk']['File'],
            "repository": "arcadia",
            "system": "arcadia",
        } for x in data['Chunks']]

    @staticmethod
    def _get_file(arcadia_path, row):
        file = row['file']
        system = row['system']
        file_path = os.path.join(arcadia_path, file)

        if not os.path.isfile(file_path):
            logger.info("File does not exist in arcadia: %s", file_path)
            return None

        record = {
            "url": None,
            "path": file,
            "last_commit_time": None,
            "vcs_type": system,
            "project": file.rsplit("/", 1)[0],
        }

        with open(file_path, mode="r") as package_file:
            record["file"] = unicode(package_file.read())

        for lock_variant in ("package-lock.json", "yarn.lock"):
            lock_path = file_path.replace("package.json", lock_variant)

            if os.path.isfile(lock_path):
                with open(lock_path, mode="r") as lock_file:
                    record["lock_file"] = unicode(lock_file.read())
                    record["lock_variant"] = lock_variant
                break
            else:
                record["lock_file"] = None
                record["lock_variant"] = None

        return record
