# -*- coding: utf-8 -*-

import logging
import io

from sandbox import sdk2
from sandbox.projects.oko import OkoCrawlerResult

import boto3
from botocore.exceptions import ClientError

logger = logging.getLogger(__name__)


class ResourceUploader(object):
    def __init__(self, task):
        self.task = task

    def upload(self, files):
        logger.debug("Uploading files to resources")

        resource = OkoCrawlerResult(
            task=self.task,
            description="Package files for Oko from {}".format(self.task.Context.date),
            path="files",
            ttl=10,
        )

        resource_data = sdk2.ResourceData(resource)
        for file in files:
            self._add_file_to_resource(file, resource_data)

        resource_data.ready()
        logger.info("Uploaded files to resource!")

    def _add_file_to_resource(self, file, resource_data):
        files_dir = self._make_file_dir(file, resource_data)
        files_dir.joinpath("package.json").write_text(file["file"])
        if file["lock_file"] is not None and file["lock_variant"] is not None:
            files_dir.joinpath(file["lock_variant"]).write_text(file["lock_file"])

    def _make_file_dir(self, file, resource_data):
        path_without_file = "/".join(file["path"].split("/")[:-1])

        dir_path = resource_data.path.joinpath(
            self.task.Context.date,
            file["vcs_type"],
            file["project"],
            path_without_file,
        )
        dir_path.mkdir(parents=True, exist_ok=True)
        return dir_path


class S3Uploader(object):
    def __init__(self, task):
        self.task = task
        self.s3_token = self.task.Parameters.s3_token
        logger.debug("Creating client for host %s", str(self.s3_token.data()["Host"]))
        logger.debug("Creating client for host %s", str(self.s3_token.data()["Host"])[:])
        self.client = boto3.session.Session(
            aws_access_key_id=self.s3_token.data()["AccessKeyId"],
            aws_secret_access_key=self.s3_token.data()["AccessSecretKey"],
        ).client(service_name="s3", endpoint_url=self.s3_token.data()["Host"])

    def upload(self, files):
        logger.debug("Uploading files to S3")

        for file in files:
            dir = self._make_file_dir(file)
            path = "/".join((dir, "package.json"))
            self._upload_file(file["file"], path)
            if file["lock_file"] is not None and file["lock_variant"] is not None:
                lock_path = "/".join((dir, file["lock_variant"]))
                self._upload_file(file["lock_file"], lock_path)

        logger.info("Uploaded files to S3!")

    def _make_file_dir(self, file):
        path_without_file = "/".join(file["path"].split("/")[:-1])

        return "/".join(
            (
                item
                for item in (
                    self.task.Context.date,
                    file["vcs_type"],
                    file["project"],
                    path_without_file,
                )
                if item
            )
        )

    def _upload_file(self, file, path):
        with io.BytesIO(bytes(file)) as data:
            try:
                self.client.upload_fileobj(data, "oko-files", path)
            except ClientError as e:
                logger.error(e)
