# -*- coding: utf-8 -*-

import textwrap

import lxml.etree
import requests
from sandbox.common.errors import TaskError
from sandbox.projects.common.build import YaPackage
from sandbox import sdk2
from sandbox.sandboxsdk import parameters, svn, channel


class MdsUploadParameter(parameters.SandboxBoolParameter):
    name = 'mds_upload'
    description = 'Upload to MDS'
    required = False
    default_value = True


class OopsAgentResource(sdk2.Resource):
    """
    OOPS agent resource description
    """

    releasable = True
    executable = False
    auto_backup = True
    releasers = ['OOPS']
    release_subscribers = []


class AssembleOopsAgent(YaPackage.YaPackage):
    """
    Build oops-agent as a skycore service
    """

    type = 'ASSEMBLE_OOPS_AGENT'
    release_to = ['oops-dev@yandex-team.ru', 'skynet@yandex-team.ru']

    class ServiceVersion(parameters.SandboxStringParameter):
        name = 'service_version'
        description = 'Version to build'
        required = True

    input_parameters = [
        YaPackage.parameters.ArcadiaUrl,
        ServiceVersion,
        YaPackage.parameters.Musl,
        YaPackage.parameters.DoNotRemoveResources,
        MdsUploadParameter,
    ]

    execution_space = 4000

    def on_enqueue(self):
        self.ctx[YaPackage.parameters.BuildType.name] = YaPackage.RELEASE
        self.ctx[YaPackage.PackageTypeParameter.name] = YaPackage.TARBALL
        self.ctx[YaPackage.PackagesParameter.name] = "infra/oops-agent/build.json"
        self.ctx[YaPackage.RunTestsParameter.name] = False
        self.ctx[YaPackage.ResourceTypeParameter.name] = OopsAgentResource.name
        self.ctx[YaPackage.PublishPackageParameter.name] = False

        self.ctx[YaPackage.CompressPackageArchiveParameter.name] = True
        self.ctx[YaPackage.UseNewFormatParameter.name] = True
        self.ctx[YaPackage.parameters.CheckoutParameter.name] = True

        version = self.ctx[AssembleOopsAgent.ServiceVersion.name]
        self.ctx[YaPackage.CustomVersionParameter.name] = version
        self.ctx[YaPackage.PackageResourceDescriptionParameter.name] = {
            "infra/oops-agent/build.json": "oops-agent-%s" % (version,)}

        YaPackage.YaPackage.on_enqueue(self)

    def mds_upload(self, path, store_path):
        base_uri = 'http://storage-int.mds.yandex.net:1111'

        get_uri_http = 'http://storage-int.mds.yandex.net/get-oops'
        get_uri_https = 'https://storage-int.mds.yandex.net/get-oops'

        auth = self.get_vault_data('OOPS', 'oops_mds_auth')

        headers = {
            'Authorization': 'Basic ' + auth,
        }

        with open(path, 'rb') as fp:
            response = requests.get(base_uri + '/upload-oops/' + store_path, headers=headers, data=fp)

        if response.status_code == 200:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            uploaded_key = tree.get('key')

            if uploaded_key:
                return {
                           'http': get_uri_http + '/' + uploaded_key,
                           'https': get_uri_https + '/' + uploaded_key,
                       }, uploaded_key

            return False
        elif response.status_code == 403:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            key = tree.find('key').text

            if key:
                raise Exception('Unable to upload to mds: key already exists: %s' % (key,))
            else:
                raise Exception('Unable to upload to mds: got 403 without key')
        elif response.status_code == 401:
            raise Exception('Unable to upload to mds: perm denied (code 401)')

        raise Exception('Unable to upload to mds: code %d' % (response.status_code,))

    def on_release(self, additional_parameters):
        resources = channel.channel.rest.list_resources(task_id=self.id,
                                                        resource_type=self.ctx[YaPackage.ResourceTypeParameter.name])
        if not resources or len(resources) > 1:
            raise TaskError("Exactly one OopsAgentResource resource expected")

        resource = resources[0]
        file_name = resource.file_name

        resource_path = self.sync_resource(resource.id)

        upload_to_mds = self.ctx.get(MdsUploadParameter.name)
        if upload_to_mds:

            info = self.mds_upload(
                resource_path,
                'oops-agent/' + file_name
            )
            if not info:
                raise Exception('Unable to upload to MDS')

            mds_uri, mds_key = info

            channel.channel.sandbox.set_resource_attribute(resource.id, 'mds_key_service', 'skynet:' + mds_key)

            for typ, uri in mds_uri.items():
                channel.channel.sandbox.set_resource_attribute(
                    resource.id, '%s_mds' % (typ,), uri
                )
                channel.channel.sandbox.set_resource_attribute(
                    resource.id, '%s_mds_weight' % (typ,), 50 if typ == 'http' else 51
                )

        channel.channel.sandbox.set_resource_attribute(resource.id, 'version',
                                                       self.ctx[AssembleOopsAgent.ServiceVersion.name])
        channel.channel.sandbox.set_resource_attribute(resource.id, 'svn_url',
                                                       self.ctx[YaPackage.parameters.ArcadiaUrl.name])

        for scheme in 'http', 'https':
            channel.channel.sandbox.set_resource_attribute(
                resource.id,
                scheme + '_sandbox_proxy',
                scheme + '://proxy.sandbox.yandex-team.ru/%d' % (resource.id,)
            )
            channel.channel.sandbox.set_resource_attribute(
                resource.id,
                scheme + '_sandbox_proxy_weight',
                10 if scheme == 'http' else 11
            )

        YaPackage.YaPackage.on_release(self, additional_parameters)

    @property
    def release_template(self):
        revision, tag, branch = self.arcadia_info()

        if self.ctx['service_version']:
            subject = 'oops-agent v%s' % (self.ctx['service_version'],)
        elif tag:
            subject = 'oops-agent v%s' % (tag,)
        else:
            subject = 'oops-agent v%s@%s' % (branch, revision)

        return self.ReleaseTemplate(
            ['oops@yandex-team.ru'],
            subject,
            textwrap.dedent('''\
                Changelog:
                  ** Bugs
                     * OOPS-XXX: bug #1
                  ** Features
                     * OOPS-YYY: feature #1
            '''),
            ['stable']
        )

    def arcadia_info(self):
        """ Return revision, tagname and branchname for release form. """

        parsed_url = svn.Arcadia.parse_url(self.ctx[YaPackage.parameters.ArcadiaUrl.name])
        path = parsed_url.path
        revision = parsed_url.revision
        if path.startswith('arc/branches/oops-agent'):
            branch = path.split('/')[3]
        else:
            branch = None

        if path.startswith('arc/tags/oops-agent'):
            tag = path.split('/')[3]
        else:
            tag = None

        return revision, tag, branch


__Task__ = AssembleOopsAgent
