import os
import tarfile

from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.sdk2.helpers import subprocess as sp


class OrgthiefBin(sdk2.Resource):
    releasable = False
    any_arch = False


class OrgthiefCategories(sdk2.Resource):
    releasable = False
    any_arch = False

    country = sdk2.Attributes.String('country', required=True)
    version = sdk2.Attributes.Integer('version', required=True)


class OrgthiefBoundsList(sdk2.Resource):
    releasable = False
    any_arch = False

    tag = sdk2.Attributes.String('tag', required=True)


class OrgthiefTask(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Common parameters") as common_block:
            cluster = sdk2.parameters.String(
                'Cluster the script will work with',
                description='Example: `hahn`, `freud`, `banach`.',
                required=True,
            )
            domain = sdk2.parameters.String('Domain', description='Example: `ru`, `de`.', required=True)
            output_directory = sdk2.parameters.String('Output directory', required=True)
            user_slots = sdk2.parameters.Integer('User slots', default=15, required=True)

        with sdk2.parameters.Group("Task specific parameters") as task_block:
            with sdk2.parameters.String('Task type', multiline=True, required=True) as task_type:
                task_type.values['regular'] = task_type.Value(value='Regular', default=True)
                task_type.values['organizations'] = task_type.Value(value='Organizations')
                task_type.values['by_ids'] = task_type.Value(value='By ids')

            with task_type.value['regular']:
                workdir_ttl = sdk2.parameters.Integer('Work directory ttl', default=14, required=True)

            with task_type.value['organizations']:
                if_not_last_categories = sdk2.parameters.Bool(
                    'Choose categories resourse \n'
                    '(Or newest categories resource for your domain will be used)',
                    default=True,
                )
                with if_not_last_categories.value[True]:
                    categories = sdk2.parameters.Resource(
                        'Categories',
                        resource_type=OrgthiefCategories,
                    )
                with sdk2.parameters.String('Reload categories', multiline=True, required=True) as reload_categories:
                    reload_categories.values['new'] = task_type.Value(value='New', default=True)
                    reload_categories.values['old'] = task_type.Value(value='Old')
                    reload_categories.values['all'] = task_type.Value(value='All')

                use_specific_streets_resource = sdk2.parameters.Bool(
                    'Use specific resource with streets',
                    default=False
                )
                with use_specific_streets_resource.value[True]:
                    streets_resource = sdk2.parameters.Resource(
                        'Streets resource',
                        resource_type=resource_types.RTYSERVER_GARDEN_EXPORT,
                        required=True
                    )
                with use_specific_streets_resource.value[False]:
                    streets_resource_region = sdk2.parameters.String('Streets resource region', required=True)

                with sdk2.parameters.String(
                    'Alphabet', multiline=True, required=True,
                    description='Sometimes notes can be on latin or cyrilic alphabet.'
                ) as alphabet_type:
                    alphabet_type.values['latin'] = task_type.Value(value='Latin')
                    alphabet_type.values['cyrillic'] = task_type.Value(value='Cyrillic', default=True)

                use_coord_rects_from_resource = sdk2.parameters.Bool(
                    'Use resource with several City rectangle coordinates',
                    default=False
                )

                with use_coord_rects_from_resource.value[True]:
                    city_rect_coords_resource = sdk2.parameters.Resource(
                        'Coordinates resource',
                        resource_type=OrgthiefBoundsList
                    )

                with use_coord_rects_from_resource.value[False]:
                    city_rect_coord = sdk2.parameters.Dict(
                        'City rectangle coordinates',
                        default={
                            'min_lat': 52.3361953,
                            'max_lat': 52.6834762,
                            'min_lon': 13.0689553,
                            'max_lon': 13.769671,
                        },
                        required=True,
                        description='You should fill all possible variants of this field.'
                    )

                restart_download = sdk2.parameters.Bool('Restart download', default=False)

            with task_type.value['by_ids']:
                input_table = sdk2.parameters.String('Input table with google ids', required=True)

    def on_execute(self):
        env = {
            'YT_TOKEN': sdk2.Vault.data('robot_thoth_token'),
            'BOTANIK_TOKEN': sdk2.Vault.data('botanik_auth_token'),
        }

        result = sdk2.Resource.find(resource_type=OrgthiefBin, state=ctr.State.READY).first()
        eh.verify(result, 'Can\'t get last binary for this task.')
        bin_resource = sdk2.ResourceData(result)

        if self.Parameters.task_type == 'organizations':
            self._make_organizations_task(env=env, bin_resource=bin_resource)
        else:
            self._make_regular_task(env=env, bin_resource=bin_resource)

    def _make_regular_task(self, env, bin_resource):
        cmd = self._build_command_line(bin_resource=bin_resource)
        with sdk2.helpers.ProcessLog(self, logger='orgthief_task') as l:
            sp.check_call(cmd, stdout=l.stdout, stderr=l.stderr, env=env)

    def _make_organizations_task(self, env, bin_resource):
        categories, country, version = self._get_categories_resourse()
        streets = self._get_streets_resourse()
        graph_data = self._get_graph_data_tar_gz(streets.path)
        eh.verify(graph_data, 'Can\'t get streets archive.')

        os.mkdir('streets')
        streets_unpack_path = os.path.join(os.getcwd(), 'streets')
        with tarfile.open(str(graph_data), 'r:gz') as tar:
            tar.extractall(streets_unpack_path)

        cmd = self._build_command_line(bin_resource=bin_resource, categories=categories, streets_path=streets_unpack_path)
        with sdk2.helpers.ProcessLog(self, logger='orgthief_task') as l:
            sp.check_call(cmd, stdout=l.stdout, stderr=l.stderr, env=env)
        if self.Parameters.reload_categories != 'old':
            self._upload_new_categories_to_sandbox(country, version)

    def _get_graph_data_tar_gz(self, posix_path):
        """It should be only one resource `graph_data.tar.gz` in the directory"""
        graph_data = posix_path.joinpath('graph_data.tar.gz')
        return None if not graph_data.exists() else graph_data

    def _get_categories_resourse(self):
        if self.Parameters.if_not_last_categories:
            result = self.Parameters.categories
        else:
            result = sdk2.Resource.find(
                resource_type=OrgthiefCategories,
                state=ctr.State.READY,
                attrs={
                    'country': self.Parameters.domain.split('.')[-1],
                },
            ).first()

            eh.verify(
                result,
                (
                    'Can\'t get last ready resource {} with domain {}.'
                    'Please, choose manually categories resource'
                ).format(OrgthiefCategories, self.Parameters.domain)
            )

        return sdk2.ResourceData(result), result.country, result.version

    def _get_streets_resourse(self):
        if self.Parameters.use_specific_streets_resource:
            result = self.Parameters.streets_resource
        else:
            result = sdk2.Resource.find(
                resource_type=resource_types.RTYSERVER_GARDEN_EXPORT,
                state=ctr.State.READY,
                attrs={
                    'garden_environment': 'production',
                    'region': self.Parameters.streets_resource_region,
                },
            ).first()
            eh.verify(
                result,
                'Can\'t get last ready resource for streets with region {}.'.format(
                    self.Parameters.streets_resource_region
                )
            )
        return sdk2.ResourceData(result)

    def _build_command_line(self, bin_resource, categories=None, streets_path=None):
        cmd = [
            str(bin_resource.path),
            '--cluster={}'.format(self.Parameters.cluster),
            '--domain={}'.format(self.Parameters.domain),
            '--yt-output-directory={}'.format(self.Parameters.output_directory),
            '--user-slots={}'.format(self.Parameters.user_slots),
            self.Parameters.task_type,
        ]

        if self.Parameters.task_type == 'regular':
            cmd.extend([
                '--workdir_ttl={}'.format(self.Parameters.workdir_ttl),
            ])
        elif self.Parameters.task_type == 'organizations':
            cmd.extend([
                '--categories={}'.format(categories.path),
                '--reload_categories={}'.format(self.Parameters.reload_categories),
                '--alphabet={}'.format(self.Parameters.alphabet_type),
                '--rd_nm={}'.format(os.path.join(streets_path, 'rd_nm')),
                '--rd_el={}'.format(os.path.join(streets_path, 'rd_el')),
                '--rd_rd_el={}'.format(os.path.join(streets_path, 'rd_rd_el')),
            ])
            if self.Parameters.use_coord_rects_from_resource:
                cmd.extend([
                    '--bounds_list',
                    str(sdk2.ResourceData(self.Parameters.city_rect_coords_resource).path)
                ])
            else:
                cmd.extend([
                    '--bounds',
                    str(self.Parameters.city_rect_coord['min_lat']),
                    str(self.Parameters.city_rect_coord['max_lat']),
                    str(self.Parameters.city_rect_coord['min_lon']),
                    str(self.Parameters.city_rect_coord['max_lon'])
                ])
            if self.Parameters.restart_download:
                cmd.append('--restart-download')
        elif self.Parameters.task_type == 'by_ids':
            cmd.extend([
                '--yt-input-table={}'.format(self.Parameters.input_table)
            ])
        return cmd

    def _upload_new_categories_to_sandbox(self, country, version):
        """
        As a result of working binary create file `categories.txt`,
        that should be uploaded to sandbox for another future usages and experiments.

        When the task is completed, the resource will automatically load to sandbox.
        All you need is to create it.
        """
        OrgthiefCategories(self, 'Categories', 'categories.txt', country=country, version=version + 1)
