from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.orgthief.tycoon import OrgthiefTycoonBin
from sandbox.sdk2.helpers import subprocess as sp


class TycoonTask(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        cluster = sdk2.parameters.String(
            'Cluster',
            description='Example: `hahn`, `freud`, `banach`.',
            required=True,
            default='hahn',
        )
        output_directory = sdk2.parameters.String(
            'Output directory',
            description=(
                'Directory, in which all new tables with '
                'name as date and fields `rank`, `pageUrl`, `id`, `date` will be saved'
            ),
            required=True,
            default='//home/extdata/tycoon/bucket',
        )
        serps_table = sdk2.parameters.String(
            'Serps table path',
            description='Path to table with data from serps loads',
            required=True,
            default='//home/sprav/assay/metrics/ScraperSerpsParsed',
        )
        serps_modification_time_file = sdk2.parameters.String(
            'Serps modification time file',
            default='//home/extdata/tycoon/serps_modification_time',
            required=True,
        )

        with sdk2.parameters.Group("Serps table specific parameters") as serps_block:
            cron_ids = sdk2.parameters.List(
                'Cron ids',
                default=[4852, 6568, 8220, 8912],
                required=True,
            )
            max_rank = sdk2.parameters.Integer(
                'Max rank',
                default=5,
                required=True,
            )

    def on_execute(self):
        env = {
            'YT_TOKEN': sdk2.Vault.data('robot_thoth_token'),
        }

        result = sdk2.Resource.find(resource_type=OrgthiefTycoonBin, state=ctr.State.READY).first()
        eh.verify(result, 'Can\'t get last binary for this task.')
        bin_resource = sdk2.ResourceData(result)

        cmd = self._build_command_line(bin_resource=bin_resource)
        with sdk2.helpers.ProcessLog(self, logger='tycoon_task') as l:
            sp.check_call(cmd, stdout=l.stdout, stderr=l.stderr, env=env)

    def _build_command_line(self, bin_resource):
        cmd = [
            str(bin_resource.path),
            '--cluster={}'.format(self.Parameters.cluster),
            '--output_yt_directory={}'.format(self.Parameters.output_directory),
            '--serps_data_yt_table={}'.format(self.Parameters.serps_table),
            '--serps_modification_time_file={}'.format(self.Parameters.serps_modification_time_file),
            '--max_rank={}'.format(self.Parameters.max_rank),
            '--cron_ids',
        ]
        for elem in self.Parameters.cron_ids:
            cmd.append(str(elem))
        return cmd
