import requests
import socket
import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.projects.common import decorators
from sandbox.projects.ott.backend.common.arcanum import ArcanumClient
from sandbox.projects.ott.backend.common.teamcity import TeamCityClient

import logging


class OttTeamcityPrBuild(sdk2.Task):
    """Trigger TeamCity build for PR"""

    class Parameters(sdk2.Parameters):
        teamcity_token_vault_name = sdk2.parameters.String('Vault item with token for TeamCity',
                                                           default='ott-infra-teamcity-token')
        arcanum_token_vault_name = sdk2.parameters.String('Vault item with token for Arcanum',
                                                          default='ott-infra-arcanum-token')
        build_conf_id = sdk2.parameters.String('TeamCity build configuration id',
                                               default='MediaServices_Ott_OttBackend_MonorepoDeployUnifiedArcadia')
        param_review_id = sdk2.parameters.Integer('Arcanum review id')
        param_zipatch = sdk2.parameters.String('Patch url')
        param_testenv_task_id = sdk2.parameters.String('TestEnv task id')
        serviceName = sdk2.parameters.String('serviceName', default = '')
        sendOnFailure = sdk2.parameters.Bool('sendOnFailure', default = False)
        description = 'wait'

    class Requirements(sdk2.Requirements):
        disk_space = 1024
        cores = 1  # exactly 1 core
        ram = 1024  # 1GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    @property
    @common.utils.singleton
    def teamcity_client(self):
        teamcity_token = sdk2.Vault.data(self.Parameters.teamcity_token_vault_name)
        return TeamCityClient(auth = teamcity_token)

    @property
    @common.utils.singleton
    def arcanum_client(self):
        arcanum_token = sdk2.Vault.data(self.Parameters.arcanum_token_vault_name)
        return ArcanumClient(auth=arcanum_token)

    @decorators.retries(5, delay=0.5, exceptions=(requests.RequestException, socket.error))
    def trigger_build(self, review_id, testenv_task_id):
        return self.teamcity_client.trigger_build(self.Parameters.build_conf_id, params={
            'arcanum.pr.id': review_id, 
            'testenv.task.id': testenv_task_id,
            'reverse.dep.*.arcanum.pr.id': review_id,
            'reverse.dep.*.testenv.task.id': testenv_task_id
        })

    @decorators.retries(5, delay=0.5, exceptions=(requests.RequestException, socket.error))
    def build_details(self, build_id):
        return self.teamcity_client.build_details(build_id)

    @decorators.retries(5, delay=0.5, exceptions=(requests.RequestException, socket.error))
    def cancel_queue(self, build_id, task_id):
        return self.teamcity_client.cancel_queue(build_id, task_id)

    @decorators.retries(5, delay=0.5, exceptions=(requests.RequestException, socket.error))
    def cancel_running(self, build_id, task_id):
        return self.teamcity_client.cancel_build(build_id, task_id)

    @decorators.retries(5, delay=0.5, exceptions=(requests.RequestException, socket.error))
    def build_list(self, build_id):
        return self.teamcity_client.build_list(build_id)

    def on_execute(self):
        with self.memoize_stage.trigger_build_step(commit_on_entrance=False):
            logging.info("Starting OttTeamcityPrBuild")
            zipatch = self.Parameters.param_zipatch or self.Context.arcadia_patch
            review_id = self.Parameters.param_review_id or self.Context.arcanum_review_id
            testenv_task_id = self.Parameters.param_testenv_task_id or self.Context.testenv_task_id
            logging.info("Review id " + str(review_id))
            logging.info("Zipatch " + zipatch)
            logging.info("TestEnv task id " + str(testenv_task_id))
            if not zipatch or not review_id:
                logging.info("Skip runnnig")
                return

            # trigger TeamCity build
            logging.info("Trigger TeamCity build")
            tree = self.trigger_build(review_id, testenv_task_id)
            build_id = tree.attrib['id']
            logging.info("TeamCity build %s started", build_id)
            self.Context.build_id = build_id

            self.Parameters.description = self.teamcity_client.build_url(build_id)
            if not self.Parameters.sendOnFailure:
                self.sendMessage('{} Build & Karate Tests for review started. Build [{}] (SandBox id: {})', review_id, build_id)
            
        with self.memoize_stage.pull_build_step(100):
            # pull build status
            try:
                build_tree = self.build_details(self.Context.build_id)
                if build_tree.attrib['state'] != 'finished':
                    logging.info("Build %s is not finished", self.Context.build_id)
                    raise sdk2.WaitTime(5 * 60)
                elif build_tree.attrib['status'] != 'SUCCESS':
                    raise Exception('Build {} failed'.format(self.Context.build_id))
            except Exception as e:
                raise common.errors.TaskError('Cannot check build {} status or build failed'.format(self.Context.build_id))   

    def on_failure(self): 
        if self.Parameters.sendOnFailure:
            review_id = self.Parameters.param_review_id or self.Context.arcanum_review_id
            build_id = self.Context.build_id
            if not review_id or review_id is None:
                raise common.errors.TaskError('Cannot send the error comment, because review id doesn\'t present') 
            if build_id or build_id is not None:
                self.sendMessage('{} Karate Tests were failed. Build[{}] (SandBox id: {})', review_id,  build_id)
            else:
                self.sendMessage('{} Karate Tests were failed. Build[NONE] (SandBox id: {})', review_id)

    def on_timeout(self):
        if self.Parameters.sendOnFailure:
            review_id = self.Parameters.param_review_id or self.Context.arcanum_review_id
            build_id = self.Context.build_id
            if not review_id or review_id is None:
                raise common.errors.TaskError('Cannot send the error comment, because review id doesn\'t present')   
            if build_id or build_id is not None:
                self.sendMessage('{} Karate Tests were stopped by timeout. Build[{}] (SandBox id: {})', review_id, build_id)
            else:
                self.sendMessage('{} Karate Tests were stopped by timeout. Build[NONE] (SandBox id: {})', review_id)
    
    def sendMessage(self, template, review_id, build_id = None):
        # send comment to Arcanum
        serviceNamePrefix = self.Parameters.serviceName or ''
        if serviceNamePrefix:
            serviceNamePrefix = serviceNamePrefix + ':'

        if build_id is None: 
            message = template.format(
                serviceNamePrefix,
                self.id
            )
        else:
            message = template.format(
                serviceNamePrefix,
                self.teamcity_client.build_url(build_id),
                self.id
            )
        self.arcanum_client.post_comment(review_id, message)


    def on_break(self, prev_status, status):         
        logging.info("Break OttTeamcityPrBuild, prev_status={}, status={}".format(prev_status, status))
        if status in ctt.Status.STOPPED and self.Context.build_id:
            logging.info("Stopping TeamCity build chain %s", self.Context.build_id)
            builds = self.build_list(self.Context.build_id)
            for build in builds:
                self.cancel_build(build.attrib['id'])

    def on_finish(self, prev_status, status):
        logging.info("Finishing OttTeamcityPrBuild, prev_status={}, status={}".format(prev_status, status))

    def cancel_build(self, build_id):
        logging.info("Stopping TeamCity build %s", build_id)
        try:
            build_tree = self.build_details(build_id)
            if build_tree.attrib['state'] == 'finished':
                logging.info("Build is already finished")
            elif build_tree.attrib['state'] == 'running':
                logging.info("Build is running")
                self.cancel_running(build_id, self.id)
            else:
                logging.info("Build is in queue")
                self.cancel_queue(build_id, self.id)
        except Exception as e:
            logging.error("Failed to cancel build {}", e)
